(* Abbreviations:
 *   CD: compiler directive
 *   TD: type declaration
 *   FD: function declaration
 *   Exp: expression
 *   Typ: type
 *   Stmt: statement
 *)

module type IteratorBasis = sig
  open Ast

  type cd_handler_result
  type td_handler_result
  type fd_handler_result
  type exp_handler_result
  type typ_handler_result
  type stmt_handler_result
  type iteration_result

  type iters = 
      {
	iterCDs: compiler_directive list -> (type_decl list * fun_decl list * cd_handler_result list);
	iterTDs: type_decl list -> (fun_decl list * td_handler_result list);
	iterFDs: fun_decl list -> (fd_handler_result list);
	iterExp: exp -> exp_handler_result;
	iterTyp: typ -> typ_handler_result;
	iterStmt: stmt -> stmt_handler_result
      }

  val handleCD: iters -> compiler_directive -> (compiler_directive list * type_decl list * fun_decl list * cd_handler_result)
  val handleTD: iters -> type_decl -> (type_decl list * fun_decl list * td_handler_result)
  val handleFD: iters -> fun_decl -> (fun_decl list * fd_handler_result)
  val handleExp: iters -> exp -> exp_handler_result
  val handleTyp: iters -> typ -> typ_handler_result
  val handleStmt: iters -> stmt -> stmt_handler_result

  val combine: cd_handler_result list -> td_handler_result list -> fd_handler_result list -> iteration_result
end;;

module type Iterator = sig
  open Ast;;

  type cd_handler_result
  type td_handler_result
  type fd_handler_result
  type exp_handler_result
  type typ_handler_result
  type stmt_handler_result
  type iteration_result

     
  val iterCDs: compiler_directive list -> (type_decl list * fun_decl list * cd_handler_result list)
  val iterTDs: type_decl list -> (fun_decl list * td_handler_result list)
  val iterFDs: fun_decl list -> (fd_handler_result list)
  val iterExp: exp -> exp_handler_result
  val iterTyp: typ -> typ_handler_result
  val iterStmt: stmt -> stmt_handler_result

  val iterProgram: program -> iteration_result
end;;

module MakeIterator (Basis:IteratorBasis): (Iterator with type cd_handler_result = Basis.cd_handler_result
						     and type td_handler_result = Basis.td_handler_result
						     and type fd_handler_result = Basis.fd_handler_result
						     and type exp_handler_result = Basis.exp_handler_result
						     and type stmt_handler_result = Basis.stmt_handler_result
						     and type iteration_result = Basis.iteration_result) = struct

  open Ast;;

  type cd_handler_result = Basis.cd_handler_result
  type td_handler_result = Basis.td_handler_result
  type fd_handler_result = Basis.fd_handler_result
  type exp_handler_result = Basis.exp_handler_result
  type typ_handler_result = Basis.typ_handler_result
  type stmt_handler_result = Basis.stmt_handler_result
  type iteration_result = Basis.iteration_result

  open Basis;;
      
  let rec iterCDs (cds:compiler_directive list):(type_decl list * fun_decl list * cd_handler_result list) =

    let rec recurse (toGo: compiler_directive list) 
      ((tds, fds, cdrs) : type_decl list * fun_decl list * cd_handler_result list): 
      (type_decl list * fun_decl list * cd_handler_result list) =
      
      match toGo with
	  [] -> (tds, fds, List.rev cdrs)
	| (cd::cds) ->
	    let (newCDs, newTDs, newFDs, result) = Basis.handleCD i cd in
	      recurse (newCDs @ cds) (tds @ newTDs, fds @ newFDs, result :: cdrs)

    in
      recurse cds ([], [], [])


  and iterTDs (tds: type_decl list):(fun_decl list * td_handler_result list) =

    let rec recurse (toGo:type_decl list) ((fds, tdrs):(fun_decl list * td_handler_result list)): (fun_decl list * td_handler_result list) =
      match toGo with
	  [] -> (fds, List.rev tdrs)
	| (td::tds) ->
	    let (newTDs, newFDs, result) = Basis.handleTD i td in
	      recurse (newTDs @ tds) (fds @ newFDs, result :: tdrs)
    in
      recurse tds ([], [])


  and iterFDs (fds: fun_decl list):(fd_handler_result list) =
    
    let rec recurse (toGo:fun_decl list) (fdrs:fd_handler_result list):(fd_handler_result list) =
      match toGo with
	  [] -> List.rev fdrs
	| (fd::fds) ->
	    let (newFDs, result) = Basis.handleFD i fd in
	      recurse (newFDs @ fds) (result :: fdrs)
    in
      recurse fds []

  and iterExp (e:exp) = Basis.handleExp i e

  and iterTyp (t:typ) = Basis.handleTyp i t

  and iterStmt (s:stmt) = Basis.handleStmt i s

  and i =

    { 
      iterCDs = iterCDs;
      iterTDs = iterTDs;
      iterFDs = iterFDs;
      iterExp = iterExp;
      iterStmt = iterStmt;
      iterTyp = iterTyp;
    }

  let iterProgram (p:program):iteration_result =
    let (tds, fds, cdrs) = iterCDs p.program_directives in
    let (fds', tdrs) = iterTDs (tds @ p.program_type_decls) in
    let (fdrs) = iterFDs (fds @ fds' @ p.program_fun_decls) in
      combine cdrs tdrs fdrs
end;;
