open Ast;;
open Cppfilebuf;;
open Precompiler;;
open PCTypes;;

exception CommandException of string;;

let do_precompile_only:bool ref = ref false;;



let filename = ref (None:string option) in
Arg.parse
  [ ("-unsafe", Arg.Clear do_typecheck, "");
    ("-trace", Arg.Set do_trace, "");
    ("-pc-only", Arg.Set do_precompile_only, "") ]
  (fun name -> filename := Some name)
  "";
Format.set_margin 75;
Format.open_box 0;
match !filename with
| None -> raise (CommandException "no file specified")
| Some filename ->
    (
     let file_buf = preprocessed_file_buf_from_filename filename in
     let lexbuf = lexbuf_from_preprocessed_file_buf file_buf in
     try
       (
	let p = Parse.program Lex.token lexbuf in
	(* if (!do_trace) then Format.printf "//TYPECHECKING\n" else (); *)
	let module Pc = Precompiler.Make( struct let info = { expHandlers = [Enum.enumExpHandler]; 
							      stmtHandlers = []; 
							      typHandlers = []; 
							      fcnHandlers = [];
							      tdHandlers = [Enum.enumTDHandler]; 
							      cdHandlers = [Modules.moduleCDHandler];
							      finalizers = [Modules.moduleFinalizer]; 
							      pcFilename = filename } end ) in
	let p' = Pc.pc p in
	if !do_precompile_only then (
	  Clayprinter.print_program p' ((Filename.chop_extension (Filename.basename filename)) ^ "-pp.clay");
	  Format.print_flush())
	else (
	  Format.printf "Syntax check complete.\n"; Format.print_flush ();
	  Typecheck.check_program p';
	  Format.printf "Type check complete.\n"; Format.print_flush ();
	  Code_gen.gen_code filename p';
	  Format.printf "Code Generation complete.\n"; Format.print_flush ())
       )
     with err ->
       (
	let rec print_error err =
	  (
           match err with
           | PosExn (_, PosExn (pos, err)) -> print_error (PosExn (pos, err))
           | PosExn (pos, err) ->
               (
		let pos = pos_from_offset file_buf pos in
		Format.printf "Error at file %s, line %d, column %d:@,"
		  (pos.filename)
		  (pos.line)
		  (pos.column);
		print_error err
               )
           | NestedExn (err1, err2) ->
               (
		(* XXX: this is kind of a hack *)
		(if err1 <> err2 then print_error err1);
		print_error err2
               )
           | TypeExn (t, err) ->
               (
		let rec f err =
		  match err with
		  | TypeExn (_, TypeExn (t, err)) -> f (TypeExn (t, err))
		  | TypeExn (t, err) ->
                      (
                       Format.printf "@[<v 2>In type:@,@[<2>";
                       print_typ VarMap.empty 0 t;
                       Format.printf "@]@]@,";
                       print_error err
                      )
		  | _ -> print_error err
		in
		Format.printf "@[<v 2>Error in type:@,@[<2>";
		print_typ VarMap.empty 0 t;
		Format.printf "@]@]@,";
		f err
               )
           | MessageExn (s, err) ->
               (
		Format.printf "%s@," s;
		print_error err
               )
           | Types.OverloadFailure exn_list ->
               (
		Format.printf "Could not resolve overloaded function call; each candidate causes an error:@[<v 2>";
		List.iter
		  (fun (name, err) ->
                    Format.printf "@,Attempting to use candidate function %s@,@[<v 2>" name;
                    print_error err;
                    Format.printf "@]")
		  exn_list;
		Format.printf "@]"
               )
           | SyntaxError s ->
               (
		let pos = pos_from_offset file_buf (Lexing.lexeme_start lexbuf) in
		Format.printf "Syntax error at file %s, line %d, column %d:@,"
		  (pos.filename)
		  (pos.line)
		  (pos.column)
               )
           | Types.KindMismatch (s, k1, k2) ->
               (
		Format.printf "@[<v 2>Kind mismatch: %s@,@[<2>" s;
		print_kind 0 k1;
		Format.printf "@]@,";
		Format.printf "  !=@,@[<2>";
		print_kind 0 k2;
		Format.printf "@]@]@,"
               )
           | Types.TypeMismatch (s, t1, t2) ->
               (
		Format.printf "@[<v 2>Type mismatch: %s@,@[<2>" s;
		print_typ VarMap.empty 0 t1;
		Format.printf "@]@,";
		Format.printf "  !=@,@[<2>";
		print_typ VarMap.empty 0 t2;
		Format.printf "@]@]@,"
               )
           | Types.ConstraintError (context_list, t) ->
               (
		Format.printf "@[<v 2>Constraint error.  The following facts are known:";
		let rec f context_list =
		  (
		   match context_list with
		   | [] -> ()
		   | (BConst true)::context_list -> f context_list
		   | (BBinary (BAndOp,
			       BCompare (BLeOp, t1a, t1b),
			       BCompare (BLtOp, t2a, t2b)))::context_list
                     when Arith.eq_int_arith t1b t2a ->
                       (
			Format.printf "@,@[<2>";
			print_int_arith VarMap.empty 0 t1a;
			Format.printf "<=";
			print_int_arith VarMap.empty 0 t1b;
			Format.printf "<";
			print_int_arith VarMap.empty 0 t2b;
			Format.printf "@]";
			f context_list
                       )
		   | (BBinary (BAndOp, t1, t2))::context_list -> f (t1::t2::context_list)
		   | t::context_list ->
                       (
			Format.printf "@,@[<2>";
			print_bool_arith VarMap.empty 0 t;
			Format.printf "@]";
			f context_list
                       )
		  ) in
		f context_list;
		Format.printf "@]@,@[<v 2>Using these facts, the solver could not conclude:@,@[<2>";
		print_bool_arith VarMap.empty 0 t;
		Format.printf "@]@]@,"
               )
           | Types.TypeError s ->
               (
		Format.printf "Type error: %s@," s
               )
           | Arith.NotFormula t ->
               (
		Format.printf "Cannot generate constraint containing:@,@[<2>";
		print_typ VarMap.empty 0 t;
		Format.printf "@]@,"
               )
           | err ->
               (
		Format.printf "%s@," (Printexc.to_string err)
               )
	  ) in
	Format.open_vbox 2;
	print_error err;
	Format.print_flush ();
	exit 1
       )
    )
;;

