(* This file figures out what code to generate for C/C++ *)

open Ast;; (* Abstract Syntax Tree *)

(* search for TODO *)

type ccode =
  | Omit
  | CStmtBlocks of (cstmt list * neccesity) list * cexp
  | CIndStmts of cstmt_raw list * cexp option
  | CExp of cexp * neccesity
and neccesity = Must | May
and cexp = cexp_raw * typ_ann option
and cexp_raw =
| CEVar of var ref
| CECall of cexp * string option * actual_param list
| CEAssign of assign_op * cexp * cexp
| CERecord of (formal_param * actual_param) list
| CEMember of cexp * int
| CEBool of bool
| CEInt of big_int
| CEOverload of string list * string option ref
| CEStruct of var
and cstmt = cstmt_raw
and cstmt_raw =
| CSDecl of var * cexp
| CSExp of cexp
and actual_param =  cexp option
and formal_param = (string * typ_ann option) option
;;

(*********************** 
 *
 * Utility functions 
 *
 ************************)

(* Globals *)

let clay_prefix = "Clay";;
let clay_datatype = clay_prefix ^ "_Obj";;
let clay_datatype_fieldname = "A";;
let temp_counter= ref 0;;
let clay_tupletype = clay_prefix ^ "_Tuple";;
let clay_tupletype_fieldname = "tuple_element";;
let tuple_sizes = ref [2];;
let c_true = "1";;
let c_false = "0";;


let int_to_string (i:int):string =  
  (Int32.to_string (Int32.of_int i))
;;

let make_new_temp_var (x:unit):var =
  temp_counter := !temp_counter + 1;
  (clay_prefix ^ "_temp_" ^(int_to_string !temp_counter),0)
;;

let var_to_string (variable:var):string = 
  let (str, num) = variable in
  if num=0 then str 
  else 
    (
     "I" ^ (int_to_string num) ^ "_" ^ str
    )
;;

let exp_to_string (e:exp):string = 
  match e.exp_raw with
  | EVar v -> "EVar"
  | EBool b -> "EBool"
  | EInt bint -> "EInt"
  | EUnit -> "EUnit"
  | ECall (ex, elist) -> "ECall"
  | EOverload (_, _) -> "EOverload"
  | EAssign (aop,e1,e2) -> "EAssign"
  | EStruct v -> "EStruct"
  | ERecord (lin,lst) ->"ERecord"
  | EMember (ex,s) -> "EMember"
  | ETApp (ex ,l) -> "ETApp"
  | EPack (ex, t, l) -> "EPack"
;;

let assign_op_to_string (ao:assign_op):string =
  match ao with
  | AssignOp -> "="
  | MultAssignOp -> "*="
  | DivAssignOp -> "/="
  | ModAssignOp -> "%="
  | AddAssignOp -> "+="
  | SubAssignOp  -> "-="
  | LShiftAssignOp  -> "<<="
  | RShiftAssignOp -> ">>="
  | BitwiseAndAssignOp -> "&="
  | XorAssignOp -> "^="
  | BitwiseOrAssignOp -> "|="
;;

let add_tuple_size (i:int):unit =
  (*Format.printf "Adding size %s\n" (int_to_string i); Format.print_flush ();*)
  if (List.exists (function x -> i = x) (!tuple_sizes)) 
  then tuple_sizes := !tuple_sizes
  else tuple_sizes := i :: !tuple_sizes;
  (*Format.printf "Now have %d tuple sizes\n" (List.length !tuple_sizes); *)
  (*Format.print_flush ()*)
;;

(*********************** 
 *
 * Printing stuff
 *
 ************************)

(* a small printing library made for this compiler, 
 * but can be re-used (hopefully)
 *)
open Codeprinter;;

(*  The output get sent to these string buffers. 
 *  Then, when the file is done, these buffers are
 *  commited in sequence.  Hence, we can write to different
 *  segments of teh file as needed. 
 *
 *  The Type Buffer will be at the top of the file and include any needed
 *  c files.
 *
 *  The Type Buffer will second from the top of the file and declare and
 *  define any types that will be needed.
 *
 *  The Annonymous Function Declaration Buffer will come next.  This 
 *  will declare any functions that were not in the original langugae
 *  but are needed in the destination language.  The definitions will be
 *  in the Annonymous Function Definition Buffer.
 * 
 *  The Function Declaration Buffer will contain definitions of functions
 *  that are defined in the original language, and the Function Definition
 *  Buffer will contain it's definitions.
 * 
 *)

(* Include Buffer *)
let include_buff = Buffer.create 1000;;
(* Type Buffer *)
 let type_buff = Buffer.create 1000;;
(* Function Declaration Buffer *)
 let fprototype_buff = Buffer.create 1000;;
(* Function Definition Buffer *)
 let fdefinition_buff = Buffer.create 1000;;
(* Annonymous Function Declaration Buffer *)
 let fanon_dec_buff = Buffer.create 1000;;
(* Annonymous Function Definition Buffer *)
 let fanon_def_buff = Buffer.create 1000;;
(* Emitted C Buffer *)
 let emitted_C_buff = Buffer.create 1000;;

(* TOOD : delete definition*)
let ccode_log_buff = Buffer.create 1000;;

(*  Here, we have linked the Format standard libarary to each
 *  of the buffers, so that they can be written to with the
 *  formatting libarary.
 *)

let include_printer = make_new_printer include_buff 105 
    "/* include C files used by compiled clay */";;
let type_printer = make_new_printer type_buff 105 
    "/* C types used by compiled clay */";;
let fprototype_printer = make_new_printer fprototype_buff 105
    "/* Declarations of user defined functions from clay */";;
let fdefinition_printer = make_new_printer fdefinition_buff 105
    "/* Definitions of user defined functions from clay */";;
let fanon_dec_printer = make_new_printer fanon_dec_buff 105
    "/* Declarations of annonymous functions not defined in clay */";;
let fanon_def_printer = make_new_printer fanon_def_buff 105
    "/* Definitions of annonymous functions not defined in clay */";;
let emitted_C_printer = make_new_printer emitted_C_buff 105
    "/* C code resulting from preprocessing or user-emitted code */";;


(* TOOD : delete definition*)
let ccode_log_printer = make_new_printer ccode_log_buff 105
    "/* cexp debugging info  */";;

let write_buffer = fun (outputfile:Unix.file_descr) -> 
  ( fun (buf:Buffer.t) -> 
    (Buffer.length buf) = 
    Unix.write outputfile (Buffer.contents buf) 0 (Buffer.length buf)
   )
;;


(* TOOD : delete function *)

let write_log log_printer (filename:string):unit =
  let outputfile:Unix.file_descr = 
    (Unix.openfile filename [Unix.O_WRONLY;Unix.O_CREAT;Unix.O_TRUNC] (6*64 + 4*8 + 4)) in
  let write_to_outfile:(Buffer.t->bool) = (write_buffer outputfile) in
  (
   if (write_to_outfile log_printer.p_buffer) 
   then ()
   else raise (InternalError "compiler line 149: could not output entire file");
   Unix.close outputfile;
  )
;;

let commit_to_file (filename:string):unit =
  let outputfile:Unix.file_descr = 
    (Unix.openfile filename [Unix.O_WRONLY;Unix.O_CREAT;Unix.O_TRUNC] (6*64 + 4*8 + 4)) in
  let write_to_outfile:(Buffer.t->bool) = (write_buffer outputfile) in
  (
   if 
     (
      (write_to_outfile type_printer.p_buffer) 
	&& (write_to_outfile fprototype_printer.p_buffer) 
	&& (write_to_outfile fanon_dec_printer.p_buffer) 
	&& (write_to_outfile include_printer.p_buffer) 
	&& (write_to_outfile emitted_C_printer.p_buffer)
	&& (write_to_outfile fdefinition_printer.p_buffer) 
	&& (write_to_outfile fanon_def_printer.p_buffer)
     ) then ()
   else raise (InternalError "compiler line 149: could not output entire file");
   Unix.close outputfile;
  )
;;
  

(*************************
 *
 * Dealing with types
 *
 *************************)

(* takes the number of words needed and returns a name of a type that will hold it *)
  let clay_datatype_of_words (words:int):string = 
    match words with
    | 0 -> "void"
    | 1 -> "unsigned long"
    | _ -> 
	(
	 let wstring = (int_to_string words) in
	 (* clay_datatype ^ "<" ^ wstring ^ ">" LW OLD *)
	 clay_tupletype ^ "_" ^ wstring (* LW NEW *)
	)
;;

(* takes the number of bits needed and returns a name of a type that will hold it *)
let clay_datatype_of_size (bits:int):string = 
  let words = bits / 32 in 
  if ((words*32) < bits)
  then clay_datatype_of_words (words+1)
  else clay_datatype_of_words words
;;

let clay_datatype_size (bits:int):int = 
  let words = bits / 32 in 
  if ((words*32) < bits)
  then words+1
  else words
;;

let int_size = 32;;
let bool_size = 32;;

let rec size_of_kind (k:kind):int =
  match k with
    K2 k2 -> size_of_kind2 k2
  | KInt -> int_size
  | KBool -> bool_size
and size_of_kind2 (k:kind2):int =
  match k with 
    KArrow (l, k2) -> size_of_kind2 k2
  |  KType (n, l) -> n
;;

let typ_ann_opt_to_string  (t_o:typ_ann option):string =
  match t_o with
    Some t_a -> 
      (
       let Typ_ann(_,k,_) = t_a in 
       clay_datatype_of_size (size_of_kind k)
      )
  | None -> raise (InternalError "compiler line 202: typechecker passed empty type annotation")
;;

let size_of_typ_ann_opt (t_o: typ_ann option):int =
  match t_o with
    Some t_a -> let Typ_ann(_,k,_) = t_a in size_of_kind k 
  | None -> raise (InternalError "compiler line 208: typechecker passed empty type annotation")
;;



(* For each tuple size, n, generate 
 *   typedef struct {
 *     unsigned long tuple_element_0;
 *     unsigned long tuple_element_1; 
 *     ...
 *     unsigned long tuple_element_(n-1);
 *   } Clay_tuple_n;
 *
 *)

let rec make_subrange_list (start:int)(finish:int):(int list) = 
  if (start = finish) 
  then [start]
  else start :: (make_subrange_list (start+1) finish)
;;

let rec gen_tuple_element (size:int):unit =
  print_newline ();
  print_string ((clay_datatype_of_words 1) ^ " tuple_element_" ^ (int_to_string size) ^ ";");
;;

let gen_tuple_elements (size:int):unit =
  List.iter gen_tuple_element (make_subrange_list 0 (size-1))
;;

let gen_tuple_type (size:int):unit = 
  (*Format.printf "  tuple size %d\n" size; Format.print_flush ();*)
  print_string  "typedef struct {";
  start_block ();
  gen_tuple_elements size;
  end_block ();
  print_string  ("} " ^ clay_tupletype ^ "_" ^ (int_to_string size) ^ ";\n");
  print_newline ();
;;

let gen_tuple_types (sizes:int list):unit =
  (*Format.printf "Generating %d different tuples.\n" (List.length sizes); Format.print_flush ();*)
  List.iter gen_tuple_type sizes
;;
 
(* Might need to be redone *)
 let gen_types (types:type_decl list):unit = 
   (
    set_printer type_printer;
    print_string "/* in type generator */";
    print_newline ();
    print_newline ();
    print_string "#ifndef __CLAY_TYPE__";
    print_newline ();
    print_string "#define __CLAY_TYPE__";
    print_newline ();
    print_newline ();

(*   print_string "template <int n>";
     print_newline ();
     print_string ("struct " ^ clay_datatype ^ " {");
     print_newline ();
     print_string ("  unsigned long " ^ clay_datatype_fieldname  ^ "[n];");
     print_newline ();
     print_string ("  unsigned long& operator[](int i) {return " ^ clay_datatype_fieldname  ^ "[i];}");
     print_newline ();
     print_string ("};");
     LW OLD *)
    gen_tuple_types(!tuple_sizes);
    print_newline ();
    print_newline ();
    print_string "#endif  /* #IFNDEF __CLAY_TYPE__ */";
    print_newline ();
    print_newline ();
    revert_printer ();
   )
;;

(**************************************
 *
 * Utility functions 
 * for zero-sized ommisions:
 *
 * C Expressions
 *
 ***************************************)

(* This function takes a string and searches a list for it.
 * It returns the integer position of the string in the list,
 * where 0 indicates that it was the head. To be more specific, 
 * the integer indicates the position of the first occourance of 
 * the string in the list that is not of size zero.  If no
 * such string exists in the list, it raises an error.
 *)
let rec search_field_list
    (field_list_opt:(string * typ_ann) list option) (s:string):int = 
  match field_list_opt with
  | None ->  raise (InternalError 
		      "compiler line 283: missing type annotation for fields")
  | Some field_list -> search_field_list_helper 0 field_list s
and search_field_list_helper (count:int)
    (field_list:(string * typ_ann) list) (s:string):int =
  match field_list with
  | [] -> raise (InternalError 
		   ("compiler line 391: can not get a non-existing member field: " ^ s))
  | (str,t)::fl ->
      (
       let sz = (size_of_typ_ann_opt (Some t)) in
       match sz with
	 (*CKH commented out: | 0 -> search_field_list_helper count fl s *)
       | 0 | 32 ->
	   (
            if (str=s) 
            then count
            else search_field_list_helper (count+(sz/32)) fl s
	   )
       | _ -> raise (InternalError 
		       "compiler line 301: member field of size other than 0 or 32 bits")
      )
;;

(************************************************************************** 
 *
 *  EXPLANATION of C Expression and the algorithm that
 *  creates them
 *
 * 
 *  This is the most complicatated part of the code generator
 *  so I will explain it here.
 *
 *  The Clay Lanugage allows for expressions and data which 
 *  consists of exactly zero bits.  Obviously, such things do
 *  not exist in C++ or C.  At first glance, the appropriate action
 *  may seem to be to omit any expression of size zero.  This, 
 *  however, is uncorrect.  Suppose, for example, that a sub expression
 *  of this zero-sized expression is a function call which writes to 
 *  memory.  Failing to call this function will have serious consequences
 *  in the run-time state of the program.
 *
 *  While considering this problem, another concern arose.  Clay makes
 *  certain guarantees about the order of evaluation of sub expressions for 
 *  some of it's expressions where C++ does not.  For example, in Clay, 
 *  in the expression ECall(ex,ex_list), the name of the function (ex)
 *  is evaluated first, and then each parameter from left to right (ex_list
 *  from head to tail).  This is not the case in C++.  However, we can cheat
 *  and allow subexpressions to be evaluated in any order if none of them 
 *  have side-effects, or in other words, affect the state of the program.
 *
 *  So, the only time when we need to take action is when an expression has
 *  a subexpression which affects the state of the program.  When we have
 *  such an expression, the first thing we need to do is look at all the 
 *  sub-expressions, and determine which ones do not affect the state or
 *  are not affected by it (in other words, constants), and those sub-expressions
 *  which might do either.  
 *
 *  We will need to construct a block of stamtements which will ensure that
 *  the non-constant sub-expressions will be evaluated in order.  To be more
 *  specific, for each non-constant subexpression, we will assign a temporary
 *  variable to that sub-expression, and then evaluate the expression with the
 *  temp variable in the place of the original subexpression.  For example,
 * 
 *  For example, ECall(EVar(ref ("f",0)),[(EInt(1));(EInt(1))])
 * 
 *  would generate (f)(1,1)
 *
 *  but ECall(EVar(ref ("f",0)),[ECall(EVar(ref ("f",0)),[(EInt(1));(EInt(1))]);(EInt(1))])
 *
 *  would generate
 *
 *  unsigned long Clay_temp_1 = f;
 *  unsigned long Clay_temp_2 = (f)(1,1);
 *  (Clay_temp1)(Clay_temp_2,1)
 *  
 *  What happened to EVar(ref ("f",0)) and ECall(EVar(ref ("f",0)),[(EInt(1));(EInt(1))])
 *  here was what I will call "bumping it to a (statement) block" above the expression.
 *
 *  If a sub-expression of some expression changes the state of the program, then
 *  all non-constant sub-expressions must be bumped.
 *
 *  So, what needs to happen in the code generator is that all expressions must be
 *  translated into c runtime expressions and/or statement blocks.  We have made
 *  several types (see ast.ml) and functions to do this.
 *
 *  These c runtime expresions (the type is ccode) come in four forms, but before enumerating 
 *  them, there is something else to mention (ok, maybe this is poorly written and tacky,
 *  but these are comments, not a novel).  All the ccode's have a property of having one
 *  of three levels of necessity: must omit, must evaluate, or may do either.  A ccode that 
 *  is of size zero must be omitted.  A ccode which causes side effects must be evaluated.
 *  Any other ccode may go either way:  its value is important for whatever it is 
 *  contained in.  In other words if a ccode has a subexpression that may be evaluated,
 *  if expression containing it must be omitted, this subexpression will be omitted.  If the
 *  expression containing it must be evaluated, this subexpression must also be evaluated.
 *
 *  The four type constructors associated with ccode are
 *  Omit - this is the placeholder for omitted expressions
 *  CExp - this takes three fields: an expression, it's type, and necessity, which
 *         is May (for "may be evaluated or may be omitted") or Must (for "must be evaluated")
 *  CIndStmts - this represents expressions which are "bumped to a statment block".  It contains
 *                 the list of statments, and a optional handle.  The handle is a cexp which
 *                 contains the value which the bumped expression would have had.  The handle must
 *                 be safe to omit (not causing side effects) and insensitive to order of evaluatation.
 *                 Hence, it must either be a constant, or a temporary variable intorduced by the 
 *                 code generator.  This type constructor has no neccesity field because it is always
 *                 Must: every statement in the block is neccesarry, and the handle is always May.
 *                 In the earlier example, the declarations of Clay_temp_1 and Clay_temp_2 would 
 *                 be in the statement blocks, and the handle would be Clay_temp_2.
 *  CStmtBlocks - this contains multiple statement blocks, which have a neccesity tag, and a handle.
 *                   the reason for this would be best understood if I know show the general algorithm
 *                   for making ccode's from an expression.
 *
 *  To make a ccode from an expression e:
 *
 *  If e must be omitted: bump only those subexpressions which must be evaluated (in other words, 
 *                        those which cause side effects) to a statement block.  return a 
 *                        CIndStmts with this block, but no handle.
 *  If e may be evaluated: If no subexpressions cause side-effects, then stuff e into CExp and give
 *                         it the May tag.  If some do, bump up all sub expressions, in order, as follows:
 *                         if the subexpression must be ommited, then just skip it.  If it is a "may", then
 *                         bump it to a statment block and give that block a May tag.  If it must be 
 *                         evaluated, bump it to a statement block and give that block a Must tag. after
 *                         cocatenating these blocks, add on more to the end: bump e to a block, and for 
 *                         its sub expressions, use the handles from it's bumped sub-expressions. The handle
 *                         from bumping e should be the handle for the CStmtBlocks.
 *  If e must be evaluated: If no subexpressions cause side-effects, then stuff e into CExp and give
 *                          it the Must tag.  If some do, bump up all sub expressions, in order, as follows:
 *                          if the subexpression must be ommited, then just skip it.  "may" and "must" subexpressions
 *                          need to be evaluated, so they can all be bumped to a statment block and given Must 
 *                          tags.  Then, these blocks should be cocatenate into one big block with one tag (Must)
 *                          with e bumped and cocatenated to the end. The handle from bumping e should be the handle 
 *                          for the CIndStmts.
 *
 *  A property here we notice is that if e is "must" then all "may" subexpressions become "must".  similarly, if
 *  e is "omit", then all  "may" subexpressions become "omit".
 *
 *  You will also notice that there is nothing like CIndStmts for a block of May expressions.  The reason is this:
 *  The only time sub-expressions get bumped are when this or some other sub-expression has a side effect.  Hence, when
 *  we cocatenate these blocks, at least one will be a Must.  Hence, we have CStmtBlocks, which allow for mutiple blocks
 *  each with their own necessity tag.
 *
 *  With the above algorithm, we have used the following chart to determine the neccesity of various expresions
 *
 *
 *     e is:    | size of e is 0 | size of e is multiple of 32 |
 *  ============|================|=============================|
 *  EVar        |      Omit      |              May            |
 *  ============|================|=============================|
 *  EBool       |  not possible  |              May            |
 *  ============|================|=============================|
 *  EInt        |  not possible  |              May            |
 *  ============|================|=============================|
 *  EUnit       |      Omit      |           not possible      |
 *  ============|================|=============================|
 *  ECall       |  if (size of ex is 0) Omit                   |
 *  (ex,e_list) |  if (size of ex is multiple of 32) Must      |
 *  ============|================|=============================|
 *  EAssign     |      Omit      |             Must            |
 *  ============|================|=============================|
 *  EOverload   |      Omit      |              May            |
 *  ============|================|=============================|
 *  EStruct     |      Omit      |              May            |
 *  ============|================|=============================|
 *  ERecord     |      Omit      |              May            |
 *  ============|================|=============================|
 *  EMember     |      Omit      |              May            |
 *  ============|================|=============================|
 *  ETApp       | whatever it is |   whatever it is for ex     |
 *  (ex,_)      |     for ex     |                             |
 *  ============|================|=============================|
 *  EPack       | whatever it is |   whatever it is for ex     |
 *  (ex,_,_)    |     for ex     |                             |
 *  ============|================|=============================|
 * 
 *
 *********************************************************************************)



(**************************************
 *
 * Functions implementing parts of
 * the C Expression creation algorithm
 *
 ***************************************)


(* TODO hand audit this whole section *)

(* This function takes the blocks inside a CStmtBlocks (or anything of the same
 * structure) and omits any block that has  a May necessity tag, and cocatenantes the
 * remaining blocks into a single statement that can be stuffed into a CIndStmts
 *
 * This is a helper function for discard_non_side_effecting_ccode 
 * which should be used instead of this where convenient.
 *
 * The correctness of this is dependent on the fact that we can prove that
 * no Must block is dependent on a May block.  This is true if no handle from
 * a May block is used in a Must block.  In other words, any temp variables declared
 * in a May block can only be used in a May block.  To make sure that this is the
 * case, inspect bump_cexps_to_CStmtBlocks and make_ccode and read the comment
 * at the end of the comments above process_sub_ccodes
 *)
let rec discard_non_side_effecting_stmt_blocks 
    (cstmt_blocks:(cstmt list * neccesity) list):(cstmt_raw list) = 
  match cstmt_blocks with
  | [] -> []
  | (some_cstmts,Must)::other_stmts -> 
      some_cstmts@(discard_non_side_effecting_stmt_blocks other_stmts)
  | (some_cstmts,May)::other_stmts ->
      discard_non_side_effecting_stmt_blocks other_stmts
;;


(* This function takes a ccode and omits anything that has 
 * a May necessity tag, and returns that which must be 
 * evaluated, in the form of a ccode.
 *)
let discard_non_side_effecting_ccode (cc:ccode):ccode =
  match cc with
  | Omit -> Omit
  | CStmtBlocks(cstmt_blocks,_) ->
      CIndStmts(discard_non_side_effecting_stmt_blocks cstmt_blocks,None)
  | CIndStmts(cstmt_list,ce_opt) -> CIndStmts(cstmt_list,None)
  | CExp(ce,Must) -> CExp(ce,Must)
  | CExp(_,May) -> Omit
;;


(* This function takes the blocks inside a CStmtBlocks (or anything of the same
 * structure) and and cocatenantes the blocks into a single statement list that 
 * can be stuffed into a CIndStmts.  Hence, all statements will now be 
 * interpretted as having a necessity of "Must"
 *
 *)
let rec make_cstmt_blocks_mandatory (cstmt_blocks:(cstmt list * neccesity) list):(cstmt_raw list) = 
  match cstmt_blocks with
  | [] -> []
  | (some_cstmts,Must)::other_stmts -> some_cstmts@(make_cstmt_blocks_mandatory other_stmts)
  | (some_cstmts,May)::other_stmts -> some_cstmts@(make_cstmt_blocks_mandatory other_stmts)
;;


(* This function takes a list of ccode's and bumps any non-constant or side-effect causing expression
 * or sub expression into a list of cstmt's.  More description on why this needs to happen and
 * some more details about it can gotten from the explanation of the c-expression creation algorithm.
 * If you want to know the details of how certain ccode's shoudl be handled, read the code:
 *)
let rec bump_cexps_to_CStmtBlocks (cc_list:ccode list):((cstmt list * neccesity) list * actual_param list) =
  match cc_list with
  | [] -> ([([],May)],[])
  | cc::tl ->  
      ( 
	let (tl_cstmts,tl_param) = (bump_cexps_to_CStmtBlocks tl) in
	match cc with
	| Omit -> (tl_cstmts,None::tl_param)
	| CStmtBlocks(cstmt_blocks,handle) -> (cstmt_blocks@tl_cstmts,Some(handle)::tl_param)
	| CIndStmts(cstmt_list,handle) -> ((cstmt_list,Must)::tl_cstmts,handle::tl_param)
	| CExp((CEVar(v),t),nec) -> 
	    ( 
	      let temp = make_new_temp_var () in
	      (([CSDecl(temp,(CEVar(v),t))],nec)::tl_cstmts,Some(CEVar(ref temp),t)::tl_param)
	     )
	| CExp((CECall(ce_fun_name,cast_opt,ce_param_list),t),nec) -> 
	    ( 
	      let temp = make_new_temp_var () in
	      (([CSDecl(temp,(CECall(ce_fun_name,cast_opt,ce_param_list),t))],nec)::tl_cstmts,Some(CEVar(ref temp),t)::tl_param)
	     )
	| CExp((CEAssign(ao,ce1,ce2),t),nec) -> 
	    ( 
	      let temp = make_new_temp_var () in
	      (([CSDecl(temp,(CEAssign(ao,ce1,ce2),t))],nec)::tl_cstmts,Some(CEVar(ref temp),t)::tl_param)
	     )
	| CExp((CERecord(f_and_a_params),t),nec) -> 
	    ( 
	      let temp = make_new_temp_var () in
	      (([CSDecl(temp,(CERecord(f_and_a_params),t))],nec)::tl_cstmts,Some(CEVar(ref temp),t)::tl_param)
	     )
	| CExp((CEMember(cex,index),t),nec) -> 
	    ( 
	      let temp = make_new_temp_var () in
	      (([CSDecl(temp,(CEMember(cex,index),t))],nec)::tl_cstmts,Some(CEVar(ref temp),t)::tl_param)
	     )
	| CExp((CEBool(b),t),nec) -> (tl_cstmts,Some(CEBool(b),t)::tl_param)
	| CExp((CEInt(bigi),t),nec) -> (tl_cstmts,Some(CEInt(bigi),t)::tl_param)
	| CExp((CEOverload(str_list,str_opt_ref),t),nec) -> (tl_cstmts,Some(CEOverload(str_list,str_opt_ref),t)::tl_param)
	| CExp((CEStruct(var_name),t),nec) -> (tl_cstmts,Some(CEStruct(var_name),t)::tl_param)
       )
;;


(* Originally, there was just a function that returned a bool that you could
 * give a ccode list and it would return true iff there was a ccode in that 
 * list which would cause some side effect.   The usual course of action after this
 * would be to extract the cexp option list from the ccode list if false * or the
 * the list of cstmt blocks if true. 
 *
 * This woudl require a lot more pattern matching in make_ccode, so I do much of that 
 * here, and rather than having a function that returns a boolean value, process_sub_ccodes
 * returns (DoesntCauseSideEffects of actual_param list) if there is no side-effect in the
 * list, and (DoesCauseSideEffects of ((cstmt list * neccesity) list * actual_param list))
 * if there is.  If there is a side effect in the ccode list, process_sub_ccodes bumps
 * the ccode's for you.  If there is not, a CExp(ce,t,_) in it becomes Some(ce) and an
 * Omit becomes a None.
 *
 * * it is possible to prove that if no sub expression causes a side effect, then
 * it must be a list of either CExp 's or Omit's.  Actually, it would be more useful 
 * and easier to prove that havining a CIndStmts or CStmtBlocks would imply that 
 * there is a side effect.
 *
 * Note: To ensure that no Must block be dependent on a May block, in terms of variable 
 * declarations, any handle (in otherwords, any element) from the actual_param list
 * from DoesCauseSideEffects must be used in a May context, or make_cstmt_blocks_mandatory
 * must be called on the cstmt blocks.  To make sure this happens, check make_ccode.
 *)

type does_sub_ccodes_cause_side_effect =
  | DoesCauseSideEffects of ((cstmt list * neccesity) list * actual_param list)
  | DoesntCauseSideEffects of actual_param list
;;


let rec process_sub_ccodes(cc_list:ccode list):does_sub_ccodes_cause_side_effect =
  let (causes_side_effects,cexp_list) = processed_sub_ccodes_helper cc_list in
  if (causes_side_effects) 
  then DoesCauseSideEffects(bump_cexps_to_CStmtBlocks cc_list)
  else DoesntCauseSideEffects(cexp_list)
and processed_sub_ccodes_helper(cc_list:ccode list):(bool * actual_param list) =
  match cc_list with
  | [] -> (false, [])
  | cc::tl -> 
      (
       let (tl_causes_side_effect,tl_ccode) =  processed_sub_ccodes_helper tl in
       match cc with
       | Omit -> (tl_causes_side_effect,None::tl_ccode)
       | CStmtBlocks(_) -> (true,[])
       | CIndStmts(_) -> (true,[])
       | CExp(_,Must) -> (true,[])
       | CExp(ce,May) -> (tl_causes_side_effect,Some(ce)::tl_ccode)
      )
;;


(*******************************************
 *
 *
 *        Make Ccode 's from Exp's
 *   
 *
 ******************************************)


(* TOOD : debugging..delete from here until 
 * "TODO end delete block 1"
 *)

(* These can be deleted...they are commented out
 * in their original place in the file, and so
 * only need to be uncommented there.
 *)

let annon_fun_num = ref 0;;

(*make and return a unique name for this function*)
let make_new_record_constr_name (x:unit):string =
  annon_fun_num := !annon_fun_num + 1;
  (clay_prefix ^ "_record_contr_" ^ 
   (int_to_string !annon_fun_num))
;;

let make_record_param (s:string):string = 
  clay_prefix ^ "_record_param_" ^ s
;;

let get_struct_constructor_name (s:string):string =
  (clay_prefix ^ "_make" ^ s)
;;

(* Preconditions: 
 * The current printer is the desired one.
 * The expression can be printed immediately (no space req.)
 *
 * Postconditions: the insertion point will not be on a new line.
 * The same printer will still be current.
 * The insertion point is not nec. preceeded by whitespace.
 * 
 * Assumptions:  This function makes many assumptions, and 
 * so will need to be changed if and when the language changes.
 * I will list the assuimptions here.  
 *
 *  - every field is of 32 or 0 bits.
 *
 * In particular, this assumption is made by  
 * gen_exp_param_list and gen_get_field and the ERecord case in gen_exp
 *
 * Also, assumes that conatining parenthesis have not been set 
 * up by the caller, so these need to be added, if this is not 
 * a base level expression such as EBool.
 *)
let rec gen_exp (e:exp):unit =
  let etyp = e.exp_typ in
  match e.exp_raw with
  | EVar {contents = v} -> 
      (
       print_string (var_to_string v); 
       print_string "<var, size:";
       print_int (size_of_typ_ann_opt etyp);
       print_string ">"
      )
  | EBool b -> if (b) then print_string c_true else print_string c_false
  | EInt bi -> print_string (Big_int.string_of_big_int bi)
  | EUnit -> print_string "/* unit value */"
  | ECall (e, elist) -> 
      (
       (* should I check and make sure that etyp isn't of size 0? *)
       (* after all, e should have type TArrow*)
       print_string "(";
       print_break ();
       print_string "(";
       gen_exp e;
       print_string ")";
       print_break ();
       print_string "(";
       gen_exp_param_list elist;
       print_string ")";
       print_string "<function call, return size:";
       print_int (size_of_typ_ann_opt etyp);
       print_string ">";
       print_break ();
       print_string ")"
      )
  | EOverload (_, fname_option) -> 
      (
       match !fname_option with
       | Some fname ->
	   (
	    print_string fname
	   )
       | None -> raise (InternalError 
			  "compiler line 659: missing overloaded function name")
      )
  | EAssign (ao,e1,e2) -> 
      (
       print_string "(";
       gen_exp e1;
       print_string ")";
       print_space ();
       print_string (assign_op_to_string ao);
       print_space ();
       print_string "(";
       gen_exp e2;
       print_string ")"
      )
  | EStruct v -> 
      (
       print_string 
	 (get_struct_constructor_name ("__struct_" ^ (var_to_string v)));
       print_string "<struct constructor>"
      )
  | ERecord (_, l) ->
      (
       (* from ast.ml :  
	* ERecord of linearity * (string * typ option * exp) list
	*
	* We don't care about this linearity here.  ERecord represents
	* the syntax in which a record is created by some expressions
	* contained in brackets.  Where ever this appears, an annonymous
	* record is created on the fly.  To compile this into C, for each
	* one of these we see, we need to  generate a new function that will
	* take these expressions as paramters and returns data that will reprsent
	* this function at runtime.  
	*)
       (* First, let's get a unique name for this function *)
       let fname = make_new_record_constr_name () in 
       (
	(* Now, we'll call it to construct our data. *)
	print_string fname;
	print_break ();
	print_string "("; 
	gen_exp_param_list 
          (
           List.map
             (fun (x:(string * typ option * exp)) ->
               let (s,t,e) = x in 
               (
                if (((size_of_typ_ann_opt e.exp_typ) = 32)||((size_of_typ_ann_opt e.exp_typ) = 0))
                then e
                else raise (InternalError 
			      "compiler line 696: record field must have 32 bits")
               ) 
             ) l
          );
	print_string ")";
	print_string "<recors constructor, size:";
	print_int (size_of_typ_ann_opt etyp);
	print_string ">"
	  (* Now that we have named and called this function, let's 
           * tell some one to declare and define this function in
           * other buffers and tell him what we know 
           *)
	  (* gen_record_constr fname l *)
       )
      )
  | EMember (e, s) ->
      (
       print_string "(";
       gen_exp e;
       print_string ")[";
       match e.exp_typ with 
       | None -> raise (InternalError 
			  "compiler line 715: missing type annotation")
       | Some Typ_ann(_, _, field_option) -> 
	   (
	    print_int (search_field_list field_option s);
	    print_string "]";
	   )
      )
  | ETApp (e, _) -> gen_exp e
  | EPack (e, _, _) -> gen_exp e
(* change this with params..also, is there a way to pass a 
 * function and combine the two?
 *)
and gen_exp_param_list (elist:exp list):unit =
  match elist with
  | [] -> ()
  | hd::tl -> 
      (
       gen_exp hd;
       gen_exp_params_helper "," tl
      )
and gen_exp_params_helper (sep:string) (elist:exp list):unit =
  match elist with
  | [] -> ()
  | hd::tl -> 
      (
       print_string sep;
       print_space ();
       gen_exp hd;
       gen_exp_params_helper "," tl
      )
;;

let stub = fun (x) -> x
;;

(* TODO end delete block 1
 *)

let get_typ_ann_opt (cc:ccode):(typ_ann option) option =
  match cc with
  | Omit -> None
  | CStmtBlocks (_, (_,t)) -> Some t
  | CIndStmts (_,None) -> None
  | CIndStmts (_,Some(_,t)) -> Some t
  | CExp((_,t),_) -> Some t
;;

(* 
 * After all the explanation above, I don't feel that
 * There is much to be said.  Must of the action here
 * is dispatched to the helper functions defined above.
 * So, if you are interested in checking if this is correct
 * read the algorithm desription and the helperd above, 
 * and look here last.
 *)
let rec make_ccode2 (e:exp):ccode =
  let t = e.exp_typ in 
  let s = size_of_typ_ann_opt t in
  match e.exp_raw with
  | EVar(var_ref) ->
      (
       match s with 
       | 0 -> Omit
       | _ -> CExp((CEVar(var_ref),t),May)
      )
  | EBool(b) ->
      (
       match s with 
       | 0 -> raise (InternalError
		       "compiler line 276: impossible to have boolean exp of size 0")
       | _ -> CExp((CEBool(b),t),May)
      )
  | EInt(bint) ->
      (
       match s with 
       | 0 -> raise (InternalError
		       "compiler line 283: impossible to have integer exp of size 0")
       | _ -> CExp((CEInt(bint),t),May)
      )
  | EUnit ->
      (
       match s with 
       | 0 -> Omit
       | _ -> raise (InternalError
		       "compiler line 291: impossible to have unit exp of size other than 0")
      )
  | ECall(ex, e_list) ->
      (
       let cc_list = List.map make_ccode2 (ex::e_list) in
       match (size_of_typ_ann_opt ex.exp_typ) with 
       | 0 -> 
	   (
	    match (process_sub_ccodes cc_list) with
            | DoesCauseSideEffects(cstmt_blocks,handles) ->
		(
		 CIndStmts(discard_non_side_effecting_stmt_blocks(cstmt_blocks),None)
		)
            | DoesntCauseSideEffects(sub_cexps) -> Omit
	   )
       | _ ->
	   (
	    match s with 
            | 0 -> 
		(
		 match (process_sub_ccodes cc_list) with
		 | DoesCauseSideEffects(cstmt_blocks,Some(handle1)::handles) ->
		     (
		      let temp = make_new_temp_var () in
		      (*let cast_str = (clay_datatype_of_size s) ^ " ( * ) (...)" in*)
		      let cast_str = "" in
		      let x = stub cast_str in
		      (*CIndStmts((make_cstmt_blocks_mandatory cstmt_blocks)@[CSExp(CECall(handle1,Some(cast_str),handles),t)],None)*)
		      CIndStmts((make_cstmt_blocks_mandatory cstmt_blocks)@[CSExp(CECall(handle1,None,handles),t)],None)

		     )
		 | DoesntCauseSideEffects(Some(sub_cexp1)::sub_cexps) ->
		     (
		      (*  TODO: this might be nec., for example, the cast is needed
		       *  for functions that are fields of struct's
		       *  I am pretty sure that the runtime type of any member of 
		       *  a struct is a one-work "unsigned long"
		       *)
		      let temp = make_new_temp_var () in
		      (*let cast_str = (clay_datatype_of_size s) ^ " ( * ) (...)" in*)
		      let cast_str = "" in
		      let x = stub cast_str in
		      (*CIndStmts([CSExp(CECall(sub_cexp1,Some(cast_str),sub_cexps),t)],None)*)
		      CIndStmts([CSExp(CECall(sub_cexp1,None,sub_cexps),t)],None)
		     )
		 | _ -> raise (InternalError
				 "compiler line 638: must have at least 1 subexpression")
		)
            | _ -> 
		(
		 match (process_sub_ccodes cc_list) with
		 | DoesCauseSideEffects(cstmt_blocks,Some(handle1)::handles) ->
		     (
		      let temp = make_new_temp_var () in
		      (*let cast_str = (clay_datatype_of_size s) ^ " ( * ) (...)" in*)
		      let cast_str = "" in
		      let x = stub cast_str in
		      (*CIndStmts((make_cstmt_blocks_mandatory cstmt_blocks)@[CSDecl(temp,(CECall(handle1,Some(cast_str),handles),t))],Some(CEVar(ref temp),t))*)
		      CIndStmts((make_cstmt_blocks_mandatory cstmt_blocks)@[CSDecl(temp,(CECall(handle1,None,handles),t))],Some(CEVar(ref temp),t))
		     )
		 | DoesntCauseSideEffects(Some(sub_cexp1)::sub_cexps) ->
		     (
		      (*  TODO: this might be nec., for example, the cast is needed
		       *  for functions that are fields of struct's
		       *  I am pretty sure that the runtime type of any member of 
		       *  a struct is a one-work "unsigned long"
		       *)
		      let temp = make_new_temp_var () in
		      (*let cast_str = (clay_datatype_of_size s) ^ " (*) (...)" in*)
		      let cast_str = "" in
		      let x = stub cast_str in
		      (*CExp((CECall(sub_cexp1,Some(cast_str),sub_cexps),t),Must)*)
		      CExp((CECall(sub_cexp1,None,sub_cexps),t),Must)
		     )
		 | _ -> raise (InternalError
				 "compiler line 653: must have at least 1 subexpression")
		)
	   )
      )   
  | EAssign(a_op,{exp_raw = EVar x; exp_typ = tx},e2)->
      (
       let cc2 = make_ccode2 e2 in
       match s with 
       | 0 -> 
	   (
	    match (process_sub_ccodes [cc2]) with
            | DoesCauseSideEffects(cstmt_blocks,handles) ->
		(
		 CIndStmts(discard_non_side_effecting_stmt_blocks(cstmt_blocks),None)
		)
            | DoesntCauseSideEffects(sub_cexps) -> Omit
	   )
       | 32 ->
	   (
	    match (process_sub_ccodes [cc2]) with
            | DoesCauseSideEffects(cstmt_blocks,Some(handle2)::[]) ->
		(
		 CIndStmts((make_cstmt_blocks_mandatory cstmt_blocks)@[CSExp(CEAssign(a_op,(CEVar x, tx),handle2),t)],Some(handle2))
		)
            | DoesntCauseSideEffects(Some(sub_cexp2)::[]) ->
		(
		 CExp((CEAssign(a_op,(CEVar x, tx),sub_cexp2),t),Must)
		)
            | _ -> raise (InternalError
			    "compiler line 783: must have 1 subexpression")
	   )
       | _-> raise (InternalError 
		      "currently clay only supports assignment for values of size 32 or 0")
      )
  | EAssign(a_op,e1,e2)-> raise (InternalError 
				   "currently clay only supports assignment to local variables")
	(*
	  let cc1 = make_ccode2 e1 in
	  let cc2 = make_ccode2 e2 in
	  match s with 
	  | 0 -> 
	  (
	  match (process_sub_ccodes [cc1;cc2]) with
          | DoesCauseSideEffects(cstmt_blocks,handles) ->
          (
          CIndStmts(discard_non_side_effecting_stmt_blocks(cstmt_blocks),None)
          )
          | DoesntCauseSideEffects(sub_cexps) -> Omit
	  )
	  | 32 ->
	  (
	  match (process_sub_ccodes [cc1;cc2]) with
          | DoesCauseSideEffects(cstmt_blocks,Some(handle1)::Some(handle2)::[]) ->
          (
          CIndStmts((make_cstmt_blocks_mandatory cstmt_blocks)@[CSExp(CEAssign(a_op,handle1,handle2),t)],Some(handle2))
          )
          | DoesntCauseSideEffects(Some(sub_cexp1)::Some(sub_cexp2)::[]) ->
          (
          CExp((CEAssign(a_op,sub_cexp1,sub_cexp2),t),Must)
          )
          | _ -> raise (InternalError
          "compiler line 783: must have 2 subexpressions")
	  )
	  | _-> raise (InternalError 
          "compiler line -652.5: currently clay only has assignment for values of size 32 or 0")
	 *)
  | EOverload(s_list, s_ref) ->
      (
       match s with 
       | 0 -> Omit
       | _ -> CExp((CEOverload(s_list,s_ref),t),May)
      )
  | EStruct(v) ->
      (
       match s with
         0 -> Omit
       | _ -> CExp((CEStruct(v),t),May)
      )
  | ERecord (lin,ste_list(*:((string * typ option * exp) list)*)) -> (*Here *)
      (
       let length = List.length ste_list in
       let cc_list 
	   = List.map 
           (fun ((s,t,e):string * typ option * exp) -> make_ccode2 e) 
           ste_list in
       let formal_params =
	 List.map2 
           (
            fun ((s,t,e):string * typ option * exp) -> 
              (
               fun cc -> 
		 match get_typ_ann_opt cc with
		 | None -> None
		 | Some(ta_o) ->
		     (
		      if ((size_of_typ_ann_opt  ta_o) = 0)
                      then None
                      else Some(make_record_param s,ta_o)
		     )
              )
           ) ste_list cc_list in
       match s with 
       | 0 -> 
	   (
	    match (process_sub_ccodes cc_list) with
            | DoesCauseSideEffects(cstmt_blocks,handles) ->
		(
		 CIndStmts(discard_non_side_effecting_stmt_blocks(cstmt_blocks),None)
		)
            | DoesntCauseSideEffects(sub_cexps) -> Omit
	   )
       | _ ->
	   (
	    match (process_sub_ccodes cc_list) with
            | DoesCauseSideEffects(cstmt_blocks,handles) ->
		(
		 let temp =  make_new_temp_var () in
		 CStmtBlocks(cstmt_blocks@[([CSDecl(temp,(CERecord(List.combine formal_params handles),t))]),May],(CEVar(ref temp),t))
		)
            | DoesntCauseSideEffects(sub_cexps) ->
		(
		 CExp((CERecord(List.combine formal_params sub_cexps),t),May)
		)
	   )
      )
  | EMember(ex,str) ->
      (
       if (s!=0&&s!=32) then 
	 raise (InternalError 
		  "compiler line 1088: currently clay only has expressions of size 32 or 0") else
	 match ex.exp_typ with 
	 | None -> raise (InternalError 
			    "compiler line 1091: missing type annotation")
	 | Some Typ_ann(_, _, field_option) -> 
	     (
	      let cc = make_ccode2 ex in
	      let index = search_field_list field_option str in
	      match s with 
              | 0 -> 
		  (
		   match (process_sub_ccodes [cc]) with
		   | DoesCauseSideEffects(cstmt_blocks,handles) ->
		       (
			CIndStmts(discard_non_side_effecting_stmt_blocks(cstmt_blocks),None)
		       )
		   | DoesntCauseSideEffects(sub_cexps) -> Omit
		  )
              | _ ->
		  (
		   match (size_of_typ_ann_opt ex.exp_typ) with
		   | 32 -> 
		       (
			match (process_sub_ccodes [cc]) with
			| DoesCauseSideEffects(cstmt_blocks,Some(handle)::[]) ->
			    (
			     let temp = make_new_temp_var () in
			     CStmtBlocks(cstmt_blocks@[([CSDecl(temp,handle)]),May],(CEVar(ref temp),t))
			    )
			| DoesntCauseSideEffects(Some(sub_cexp1)::[]) ->
			    (
			     CExp((sub_cexp1),May)
			    )
			| _ -> raise (InternalError
					"compiler line 783: must have 1 subexpressions")
		       )
		   | _  ->
		       (
			match (process_sub_ccodes [cc]) with
			| DoesCauseSideEffects(cstmt_blocks,Some(handle)::[]) ->
			    (
			     let temp = make_new_temp_var () in
			     CStmtBlocks(cstmt_blocks@[([CSDecl(temp,(CEMember(handle,index),t))]),May],(CEVar(ref temp),t))
			    )
			| DoesntCauseSideEffects(Some(sub_cexp1)::[]) ->
			    (
			     CExp((CEMember(sub_cexp1,index),t),May)
			    )
			| _ -> raise (InternalError
					"compiler line 783: must have 1 subexpressions")
		       )
		  )
	     )
      )
  | ETApp(ex,_) -> make_ccode2 ex
  | EPack(ex,_,_) -> make_ccode2 ex
;;


(**************************************
 *
 * Utility functions 
 * for structs, records, and parameters
 *
 ***************************************)

(*
  let annon_fun_num = ref 0;;

(*make and return a unique name for this function*)
  let make_new_record_constr_name (x:unit):string =
  annon_fun_num := !annon_fun_num + 1;
  (clay_prefix ^ "_record_contr_" ^ 
  (int_to_string !annon_fun_num))
  ;;

  let make_record_param (s:string):string = 
  clay_prefix ^ "_record_param_" ^ s
  ;;

  let get_struct_constructor_name (s:string):string =
  (clay_prefix ^ "_make" ^ s)
  ;;
 *)

(* Takes a (formal_param = string * typ_ann option) option) list
 * and returns the string from and position of the first element 
 * of size 32, where the head of the list is position 1, the size 
 * is calculated using the typ_ann option and the size of None is
 * judged to be 0
 *)
let rec find_first_of_size_32 (formal_params:formal_param list):(string * int) =
  find_first_of_size_32_helper 1 formal_params
and find_first_of_size_32_helper (base_index:int) 
    (formal_params:formal_param list):(string * int) =
  match formal_params with
  | [] -> raise (InternalError "compiler line 271: there was no word of size 32 in list")
  | None::tl -> find_first_of_size_32_helper (base_index+1) tl
  | Some(s,t)::tl ->
      match (size_of_typ_ann_opt t) with
      | 0 -> find_first_of_size_32_helper (base_index+1) tl
      | 32 -> (s,base_index)
      | _ -> raise (InternalError "compiler line 276: all fields must have 32 bits")
;;

(* Takes a list of formal params ans, assuming
 * that these are the parameters to a record constructor, sums the
 * size of the params, hence giving the size of the record.
 *) 
let rec record_size_of_formal_params (formal_params:formal_param list):int =
  match formal_params with
  | [] -> 0
  | None::tl -> record_size_of_formal_params tl
  | Some(_,hd_t)::tl -> (size_of_typ_ann_opt hd_t) + (record_size_of_formal_params tl)
;;

(* TODO : fix comments, and clean up
 * 
 * Preconditions: 
 * The current printer is the desired one.
 * The insertion point is at the desired insertion point,
 * but not the beggining of the line.
 *
 * Postconditions: the insertion point will not be on a new line.
 * The same printer will still be current.
 * Any opened block will be closed (the tab stack will have the 
 * same state as when this function was called)
 *
 * Parameters: param_list: a list of (string*(typ_ann option))  
 * used to assign the run-time C params to the data structure.
 * 
 * Assumptions:  This function makes many assumptions, and 
 * so will need to be changed if and when the language changes.
 * I will list the assuimptions here. this function also assumes 
 * that a call to this is not suurounded by curly brackets, 
 * and adds them if neccesary.
 *
 *  - every field is of 32 or 0 bits.
 *
 *)
let rec gen_record_or_struct_constr_body (formal_params:formal_param list):unit =
  let size = record_size_of_formal_params formal_params in
  match size with
  | 0 -> 
      (
       print_string "{";
       start_block ();
       print_string "return;";
       end_block ();
       print_string "}"
      )
  | 32 -> 
      ( 
        let (str,param_num) = find_first_of_size_32 formal_params in
        (
         print_string "{";
         start_block ();
         print_string "return";
         print_space ();
         print_string str;
         print_string ";";
         end_block ();
         print_string "}"
        )
       )
  | _ ->  
      (
       print_string "{";
       start_block ();
       print_string (clay_datatype_of_size size);
       print_space (); 
       print_string "record;";
       print_newline ();
       grscb_helper 0 formal_params;
       print_string "return record;";
       end_block ();
       print_string "}"
      )
and grscb_helper (count:int) (param_list:formal_param list):unit =
  match param_list with
  | [] -> ()
  | None::tl -> 
      ( 
	print_string "/* field of size zero omitted */";
	print_newline ();
	grscb_helper count tl
       )
  | Some(s,t)::tl -> 
      (
       match (size_of_typ_ann_opt t) with
       | 32 ->
	   (  
	      (*  print_string ("record." ^ clay_datatype_fieldname ^ "["); LW OLD *)
	      print_string ("record." ^ clay_tupletype_fieldname ^ "_"); (* LW NEW *)
	      print_int count;
	      (*  print_string "]"; LW OLD *)
	      print_space ();
	      print_string "=";
	      print_space ();
	      print_string s;
	      print_string ";";
	      print_newline ();
	      grscb_helper (count+1) tl
	     )
       | 0 -> 
	   ( 
	     print_string ("/* field " ^ s ^ " of size zero omitted */");
	     print_newline ();
	     grscb_helper count tl
	    )
       | _ -> raise (InternalError "compiler line 384: record field must have 32 bits")
      )
;;

(* check this *)

(* TODO comment *)
(* Preconditions: 
 * The current printer is the desired one.
 * The insertion point is where it needs to be 
 * (immediately after the "(" which starts a param list).
 * All params have size 0 or 32 bits (enforced w/ exception.
 *
 * Postconditions: the insertion point will be after the last 
 * character that the param list outputs(the ")" of the param 
 * list will be next).  The same printer will still be current.
 * 
 * All 0 bit params will be removed, and there will probably be comment
 * in the neighborhoof of the removed param it was removed.
 *
 * Parameters: If and only if want_typ was true, the apppropriately sized clay 
 * datatype struct will appear before the naem of each param.
 *
 * The list p is of (string * (typ_ann option)).  The string is the name
 * of the paramter, and teh typ_ann option is used to figure out the C 
 * type of the param.
 *
 * A Quick word on the fomal_param type: it is assumed that all params
 * in the formal_param list are not of size zero, and that if there
 * are params of size zero, they have been replaced by a None.  Anywhere
 * where a formal_param list is created, make sure that this is true.
 *
 *)
let rec gen_params (p:formal_param list):unit =
  match p with
  | [] -> ()
  | None::tl -> 
      (
       print_string ("/* param of size zero*/");
       gen_params_helper "" tl
      )
  | Some(hd_str,hd_typ_ann)::tl -> 
      (
       print_string ((typ_ann_opt_to_string hd_typ_ann) ^ " " ^ hd_str );
       gen_params_helper "," tl
      )
and gen_params_helper 
    (sep:string) (p:formal_param list):unit =
  match p with
  | [] -> ()
  | None::tl -> 
      (
       print_space ();
       print_string ("/*param of size zero*/");
       gen_params_helper sep tl
      )
  | Some(hd_str,hd_typ_ann)::tl -> 
      (
       print_string sep;
       print_space ();
       print_string ((typ_ann_opt_to_string hd_typ_ann) ^ " " ^ hd_str );
       gen_params_helper "," tl
      )
;; 

(******************************************
 *
 * Annonymous Record Constructor Generator
 *
 ******************************************)


(* TODO : fix comments?
 * Purpose: make the header of a function
 *
 * Preconditions: 
 * The current printer is the desired one.
 * The insertion point is on the begging of the new line.
 *
 * Postconditions: the insertion point will be after the last 
 * character that the signature needs (the ")" of the param list).
 * The same printer will still be current.
 *
 * Parameters: fname : created to be the name of this function in C
 * param_list: a list of (string*(typ_ann option)) such as is 
 * required by gen_params
 * 
 * Assumptions:  This function makes many assumptions, and 
 * so will need to be changed if and when the language changes.
 * I will list the assuimptions here.
 *
 *  - every field is of 32 or 0 bits.
 *
 *)
let gen_record_constr_sig_with_c_params (fname:string) 
    (formal_params:formal_param list):unit = 
  let size =  record_size_of_formal_params formal_params in 
  (
   add_tuple_size (clay_datatype_size size);
   print_string "inline";
   print_space_no_indent ();
   print_string (clay_datatype_of_size size);
   print_space_no_indent ();
   print_string fname;
   print_break ();
   print_string "(";
   gen_params formal_params;
   print_string ")"
  )
;;


(* TODO : re-comment?
 * 
 * Preconditions: 
 * The insertion point in the fanon_def_printer and 
 * the fanon_dec_printer is on the begging of the new line.
 *
 * Postconditions: the insertion point will be on a new line in both printers.
 * The same printer that was current prior to this call will be current after.
 * The function prototype will have one line of white space after it; 
 * the function definition will have two.
 *
 * Parameters: fname : created to be the name of this function in C
 * fomal_params: a list of (string * typ option) option.  The string is the name
 * of each field either given by the user or generated by the typechecker.
 * the typ_ann option is used in this context only to compute the C runtime typ of each field.
 * it is assumed that a None in this list is an omited paramter, and hence all omitted
 * paramters are None.
 * 
 * Assumptions:  This function makes many assumptions, and 
 * so will need to be changed if and when the language changes.
 * I will list the assuimptions here.
 *
 *  - every field is of 32 or 0 bits.
 *
 *)
let gen_record_constr (fname:string) (formal_params :formal_param list):unit =
  set_printer fanon_dec_printer;
  gen_record_constr_sig_with_c_params fname formal_params;
  print_string ";";
  print_newline ();
  print_newline ();
  revert_printer ();
  set_printer fanon_def_printer;
  gen_record_constr_sig_with_c_params fname formal_params;
  print_space_no_indent ();
  gen_record_or_struct_constr_body formal_params;
  print_newline ();
  print_newline ();
  print_newline ();
  revert_printer ()
;;

(*********************** 
 *
 * Expression Generation
 *
 ************************)

(* Check this function out.  Also, see if the parens are appropriate *)


(* TODO FIx comments
 * 
 * Preconditions: 
 * The current printer is the desired one.
 * The expression can be printed immediately (no space req.)
 *
 * Postconditions: the insertion point will not be on a new line.
 * The same printer will still be current.
 * The insertion point is not nec. preceeded by whitespace.
 * 
 * Assumptions:  This function makes many assumptions, and 
 * so will need to be changed if and when the language changes.
 * I will list the assuimptions here.  
 *
 *  - every field is of 32 or 0 bits.
 *
 * In particular, this assumption is made by  
 * gen_exp_param_list and gen_get_field and the ERecord case in gen_exp
 *
 * Also, assumes that conatining parenthesis have not been set 
 * up by the caller, so these need to be added, if this is not 
 * a base level expression such as EBool.
 *)
let rec gen_cexp (ce:cexp):unit = 
  let (ce_raw,ce_t) = ce in
  match ce_raw with
  | CEVar {contents = v} -> 
      (
       match ce_t with 
       | Some(Typ_ann(TArrow _,_,_)) ->
	   (
	    print_string "(";
	    print_string (clay_datatype_of_size (size_of_typ_ann_opt ce_t));
	    print_string ")";
	    print_string (var_to_string v)
	   )
       | Some(Typ_ann(TAll _,_,_)) ->
	   (
	    print_string "(";
	    print_string (clay_datatype_of_size (size_of_typ_ann_opt ce_t));
	    print_string ")";
	    print_string (var_to_string v)
	   )
       | _ -> 
	   (
	    print_string (var_to_string v)
	   )
      )
  | CEBool b -> if (b) then print_string c_true else print_string c_false
  | CEInt bigi -> print_string (Big_int.string_of_big_int bigi)
  | CECall (cex,cast_opt,ce_list) -> 
      (
       print_string "(";
       print_break ();
       (*print_string "(";*)
       (
	match cast_opt with
	| Some(str) ->
	    (
	     print_string "(";
	     print_string str;
	     print_string ")"
	    )
	| None -> () 
       );
       (
	match cex with 
	| (CEVar {contents = v}, _) ->
	    (
	     (* Generally, circumventing our recursive procedure
	      * with this kind of analysis is bad form.  But, 
              * We have good reason.  If cex is a CEVar, then it names
	      * something that allready exists in the c runtime environment
	      * when ever this line (the one being output-ed, not this line
	      * of the code generator) gets run or called.  Now, most things
	      * in the c runtime environment have type "unsigned long" or 
	      * the templeted "struct clay_datatype".  But, v can be the name
	      * of a function declared as part of the compiled C code, and hence
	      * will have a function pointer type in the c runtime environment.
	      * If we just called "gen_cexp cex", a cast to "unsigned long" 
	      * would have been generated along with v.  Here we optomize that 
	      *cast out.
	      *)
	     print_string (var_to_string v)
	    )
	| _ -> 
	    (
	     gen_cexp cex
	    )
       );
       (*print_string ")";*)
       print_break ();
       print_string "(";
       gen_cexp_param_list ce_list;
       print_string ")";
       print_break ();
       print_string ")"
      )
  | CEOverload (_, fname_option) -> 
      (
       match !fname_option with
       | Some fname ->
	   (
	    print_string fname
	   )
       | None -> raise (InternalError 
			  "compiler line 1030: missing overloaded function name")
      )
  | CEAssign(ao,ce1,ce2) ->
      (
       print_string "(";
       gen_cexp ce1;
       print_string ")";
       print_space ();
       print_string (assign_op_to_string ao);
       print_space ();
       print_string "(";
       gen_cexp ce2;
       print_string ")"
      )
  | CEStruct v -> 
      (
       print_string 
	 (get_struct_constructor_name ("__struct_" ^ (var_to_string v)))
      )
  | CERecord l ->
      (
       (*  ERecord represents the syntax in which a record is created by some 
	* expressions contained in brackets.  Where ever this appears, an annonymous
	* record is created on the fly.  To compile this into C, for each
	* one of these we see, we need to  generate a new function that will
	* take these expressions as paramters and returns data that will reprsent
	* this function at runtime.  
	*)
       (* First, let's get our parameter lists and get a unique name for this function *)
       let (formal_params,actual_params) = List.split l in
       let fname = make_new_record_constr_name () in 
       (
	(* Now, we'll call it to construct our data. *)
	print_string fname;
	print_break ();
	print_string "("; 
	gen_cexp_param_list actual_params;
	print_string ")";
	(* Now that we have named and called this function, let's 
         * tell some one to declare and define this function in
         * other buffers and tell him what we know 
         *)
	gen_record_constr fname formal_params
       )
      )
  | CEMember (cex, i) ->
      (
       print_string "(";
       gen_cexp cex;
       print_string ")[";
       print_string (int_to_string i);
       print_string "]"
      )
and gen_cexp_param_list (ce_list:actual_param list):unit =
  match ce_list with
  | [] -> ()
  | None::tl -> 
      (
       print_string ("/*exp of size zero*/");
       gen_cexp_params_helper "" tl
      )
  | Some(ce)::tl -> 
      (
       gen_cexp ce;
       gen_cexp_params_helper "," tl
      )
and gen_cexp_params_helper (sep:string) (ce_list:actual_param list):unit =
  match ce_list with
  | [] -> ()
  | None::tl -> 
      (
       print_space ();
       print_string ("/*exp of size zero*/");
       gen_cexp_params_helper sep tl
      )
  | Some(ce)::tl -> 
      (
       print_string sep;
       print_space ();
       gen_cexp ce;
       gen_cexp_params_helper "," tl
      )
;;
  
(***********************
 *
 * CIndStmts and CStmtBlocks
 * 
 * handing statements that
 * have replaced expressions
 *
 ***********************)


(* Preconditions: 
 * The current printer is the desired one.
 * The insertion point is at the desired insertion point,
 * the beggining of the line.
 *
 * Postconditions: the insertion point will not be on a new line.
 * The same printer will still be current.
 * Any opened block will be closed (the tab stack will have the 
 * same state as when this function was called).
 *)
  let rec gen_cstmt_list  (cstmts:cstmt list):unit = 
    match cstmts with
    | [] -> ()
    | hd::[] -> 
	(
	 match hd with
	 | CSDecl(v,(ce_raw,t)) ->
	     (
	      print_string (typ_ann_opt_to_string t);
	      print_space ();
	      print_string (var_to_string v);
	      print_space ();
	      print_string "=";
	      print_space ();
	      print_string "(";
	      print_break ();
	      print_string "(";
	      print_string (clay_datatype_of_size (size_of_typ_ann_opt t));
	      print_string ")";
	      print_break ();
	      gen_cexp (ce_raw,t);
	      print_break ();
	      print_string ")";
	      print_string ";"
	     )
	 | CSExp(ce_raw,t) ->
	     (
	      gen_cexp (ce_raw,t);
	      print_string ";"
	     )
	)
    | hd::tl -> 
	(
	 match hd with
	 | CSDecl(v,(ce_raw,t)) ->
	     (
	      print_string (typ_ann_opt_to_string t);
	      print_space ();
	      print_string (var_to_string v);
	      print_space ();
	      print_string "=";
	      print_space ();
	      print_string "(";
	      print_break ();
	      print_string "(";
	      print_string (clay_datatype_of_size (size_of_typ_ann_opt t));
	      print_string ")";
	      print_break ();
	      gen_cexp (ce_raw,t);
	      print_break ();
	      print_string ")";
	      print_string ";";
	      print_newline ();
	      gen_cstmt_list tl
	     )
	 | CSExp(ce_raw,t) ->
	     (
	      gen_cexp (ce_raw,t);
	      print_string ";";
	      print_newline ();
	      gen_cstmt_list tl
	     )
	)
;;

(* TODO hand audit from here down *)

let gen_cstmt_blocks (cc_blocks:(cstmt list * neccesity) list):unit =
  List.iter (fun (cc_stmts,_) -> (gen_cstmt_list cc_stmts)) cc_blocks
;;
  
(* TODO : delete function *)


  let rec make_ccode (e:exp):ccode =
    set_printer ccode_log_printer;
    print_newline ();
    print_string "expression:";
    start_block ();
    gen_exp e;
    end_block ();
    print_string "which has size ";
    print_int (size_of_typ_ann_opt e.exp_typ);
    print_newline ();
    print_newline ();
    revert_printer ();
    let cc = make_ccode2 e in
    (
     set_printer ccode_log_printer;
     (
      match cc with
      | CExp((ce_raw,t),Must) ->
	  (
	   print_string "generates this c expression:";
	   start_block ();
	   gen_cexp (ce_raw,t);
	   end_block ();
	   print_string "which has size ";
	   print_int (size_of_typ_ann_opt t);
	   print_newline ();
	   print_string "which is of necessity Must";
	   print_newline ();
	  )
      | CExp((ce_raw,t),May) ->
	  (
	   print_string "generates this c expression:";
	   start_block ();
	   gen_cexp (ce_raw,t);
	   end_block ();
	   print_string "which has size ";
	   print_int (size_of_typ_ann_opt t);
	   print_newline ();
	   print_string "which is of necessity May";
	   print_newline ();
	  )
      | CIndStmts(cc_stmts,Some(handle,t)) ->
	  (
	   print_string "requires this block be done first:";
	   start_block ();
	   gen_cstmt_list cc_stmts;
	   end_block ();
	   print_string "and has this handle:";
	   start_block ();
	   gen_cexp (handle,t);
	   end_block ();
	   print_string "which has size ";
	   print_int (size_of_typ_ann_opt t);
	   print_newline ();
	  )
      | CIndStmts(cc_stmts,None) ->
	  (
	   print_string "requires this block be done first:";
	   start_block ();
	   gen_cstmt_list cc_stmts;
	   end_block ();
	   print_string "but has no handle:";
	   print_newline ();
	  )
      | CStmtBlocks(cc_blocks,(handle,t)) ->
	  (
	   print_string "requires these blocks be done first:";
	   List.iter (fun (cc_stmts,nec) -> 
             (
              start_block ();
              gen_cstmt_list cc_stmts;
              end_block ();
              (
               match nec with
               | Must -> print_string "which is of necessity Must"
               | May -> print_string "which is of necessity May"
              );
              print_newline ()
             )) cc_blocks;
	   print_string "and has this handle:";
	   start_block ();
	   gen_cexp (handle,t);
	   end_block ();
	   print_string "which has size ";
	   print_int (size_of_typ_ann_opt t);
	   print_newline ();
	  )
      | Omit ->
	  (
	   print_string "was omitted";
	   print_newline ();
	  )
     );
     print_newline ();
     print_string "======================================";
     print_newline ();
     revert_printer ();
     cc
    )
;;

(***********************
 *
 * Environments
 *
 ***********************)


type code_gen_env = {param_list:param list;}
;;


let make_env (plist:param list):code_gen_env =
  {param_list = plist }

(* Preconditions: 
 * The current printer is the desired one.
 *
 * Postconditions:
 * The same printer will still be current.
 *)
let rec gen_loop_var_decl (pc_list:(param*cexp) option list):unit =
  gen_loop_var_decl_helper "" pc_list
and gen_loop_var_decl_helper (sep:string) (pc_list:(param*cexp) option list):unit =
  match pc_list with 
  | [] -> ()
  | None::tl -> gen_loop_var_decl_helper sep tl
  | Some(({contents = v},_),(ce_raw,t))::tl ->
      (
       print_string (typ_ann_opt_to_string t);
       print_space ();
       print_string (var_to_string v);
       print_space ();
       print_string "=";
       print_space ();
       gen_cexp (ce_raw,t);
       print_string ";";
       print_newline ();
       gen_loop_var_decl_helper sep tl
      )
;;


let rec make_loop_var_decl (env:code_gen_env) (elist:exp list):(cstmt list * (param*cexp) option list) = 
  let plist = env.param_list in
  let ce_list = List.map make_ccode elist in
  match (process_sub_ccodes ce_list) with
  | DoesCauseSideEffects(cstmt_blocks,handles) ->
      (
       let (loop_vars:(param*cexp) option list) = 
	 List.map2 
           (
            fun (p) -> (fun (h) -> 
              (
               match h with
               | None -> None
               | Some(ce,t) -> Some(p,(ce,t))
              ))
           ) plist handles in
       ((make_cstmt_blocks_mandatory cstmt_blocks),loop_vars)
      )
  | DoesntCauseSideEffects(sub_cexps) -> 
      (
       let (loop_vars:(param*cexp) option list) = 
	 List.map2 
           (
            fun (p) -> (fun (sub_cex) -> 
              (
               match sub_cex with
               | None -> None
               | Some(ce,t) -> Some(p,(ce,t))
              ))
           ) plist sub_cexps in
       ([],loop_vars)
      )
;;


(* TODO  this is being done the wrong way: should
 * process the whole expression list, not one at 
 * a time...or should I?  maybe make two functions.
 *)

(* Preconditions: 
 * The current printer is the desired one.
 * The insertion point is on the begging of the new line.
 *
 * Postconditions: the insertion point will not be on a new line.
 * The same printer will still be current.
 * Any opened block will be closed (the tab stack will have the 
 * same state as when this function was called)
 *)
let rec gen_loop_var_update (env:code_gen_env) (elist:exp list):unit = 
  let plist = env.param_list in gen_loop_var_update_helper plist elist
and gen_loop_var_update_helper (plist:param list) (elist:exp list):unit =
  match (plist, elist) with
  | ([],[]) -> ()
  | (p::ptail, e::etail) -> 
      (
       let ({contents = v},_) = p in
       match (make_ccode e) with
       | CExp(ce,_) ->
	   (
            print_string (var_to_string  v);
            print_space ();
            print_string "=";
            print_space ();
            gen_cexp ce;
            print_string ";";
            if ((List.length etail) != 0) then print_newline ();
            gen_loop_var_update_helper ptail etail
	   )
       | CIndStmts(cc_stmts,Some(ce)) ->
	   (
            gen_cstmt_list cc_stmts;
            print_newline ();
            print_string (var_to_string  v);
            print_space ();
            print_string "=";
            print_space ();
            gen_cexp ce;
            print_string ";";
            if ((List.length etail) != 0) then print_newline ();
            gen_loop_var_update_helper ptail etail
	   )
       | CIndStmts(cc_stmts,None) ->
	   (
            gen_cstmt_list cc_stmts;
            print_newline ();
            if ((List.length etail) != 0) then print_newline ();
	    gen_loop_var_update_helper ptail etail
	   )
       | CStmtBlocks(cc_blocks,(ce)) ->
	   (
            gen_cstmt_blocks cc_blocks;
            print_newline ();
            print_string (var_to_string  v);
            print_space ();
            print_string "=";
            print_space ();
            gen_cexp ce;
            print_string ";";
            if ((List.length etail) != 0) then print_newline ();
            gen_loop_var_update_helper ptail etail
	   )
       | Omit -> gen_loop_var_update_helper ptail etail
      )
  | _ -> raise (InternalError "compiler line 149: lists need to be same length, but are not")
;;

(*********************** 
 *
 * Statement Generation
 *
 ************************)


(* Preconditions: 
 * The current printer is the desired one.
 * The insertion point is on the begging of the new line.
 *
 * Postconditions: the insertion point will not be on a new line.
 * The same printer will still be current.
 * Any opened block will be closed (the tab stack will have the 
 * same state as when this function was called)
 *)
let gen_omitted_decl (cc:ccode):unit =
  match cc with
  | CExp(ce,Must) ->
      (
       print_string "/* declaration of size zero omitted */";
       print_newline();
       gen_cexp ce
      )
  | CIndStmts(cc_stmts,_) ->
      (
       print_string "/* declaration of size zero omitted */";
       print_newline();
       gen_cstmt_list cc_stmts
      )
  | CStmtBlocks(cc_blocks,_) ->
      (
       print_string "/* declaration of size zero omitted */";
       print_newline();
       gen_cstmt_list (discard_non_side_effecting_stmt_blocks cc_blocks)
      )
  | Omit ->
      (
       print_string "/* declaration of size zero omitted */"
      )
  | CExp(ce,May) ->
      (
       print_string "/* declaration of size zero omitted */"
      )
;;


(* Preconditions: 
 * The current printer is the desired one.
 * The insertion point is on the begging of the new line.
 *
 * Postconditions: the insertion point will not be on a new line.
 * The same printer will still be current.
 * Any opened block will be closed (the tab stack will have the 
 * same state as when this function was called)
 *)
let gen_decl (v:var) (cc:ccode):unit =
  match cc with
  | CExp((ce_raw,t),_) ->
      (
       print_string (typ_ann_opt_to_string t);
       print_space ();
       print_string (var_to_string v);
       print_space ();
       print_string "=";
       print_space ();
       gen_cexp (ce_raw,t);
       print_string ";"
      )
  | CIndStmts(cc_stmts,Some(ce_raw,t)) ->
      (
       gen_cstmt_list cc_stmts;
       print_newline ();
       print_string (typ_ann_opt_to_string t);
       print_space ();
       print_string (var_to_string v);
       print_space ();
       print_string "=";
       print_space ();
       gen_cexp (ce_raw,t);
       print_string ";"
      )
  | CStmtBlocks(cc_blocks,(ce,t)) ->
      (
       gen_cstmt_blocks cc_blocks;
       print_newline ();
       print_string (typ_ann_opt_to_string t);
       print_space ();
       print_string (var_to_string v);
       print_space ();
       print_string "=";
       print_space ();
       gen_cexp (ce,t);
       print_string ";"
      )
  | _ -> raise (InternalError "compiler line 1754: can not initialize variable")
;;


(* Preconditions: 
 * The current printer is the desired one.
 * The insertion point is on the begging of the new line.
 *
 * Postconditions: the insertion point will be on a new line.
 * The same printer will still be current.
 * Any opened block will be closed (the tab stack will have the 
 * same state as when this function was called)
 * Each declaration will be own its own line.
 *
 * Paramters: l is a list of variable to be assigned, and string
 * is the name of the temp variable holding the clay datatype which
 * has the data.
 * 
 * Assumptions:  This function makes many assumptions, and 
 * so will need to be changed if and when the language changes.
 * I will list the assuimptions here.
 *
 *  - every field is of 32 or 0 bits.
 *  - every field is legal (i.e. passed the typechecker)
 *  
 * So, I will, to assign these variable, the first variable will
 * be initialized with string[0], the second with string[1], etc.  
 *
 *)
let rec gen_decls (count:int) (varlist:(var ref * typ option) list) 
    (tlist:(string * typ_ann) list) (s:string):unit =
  match (varlist,tlist) with 
  | [],[] -> ()
  | ({contents = v},_)::vtl,(_,t)::tl ->
      (
       match (size_of_typ_ann_opt (Some t)) with
       | 0 -> 
	   (
            print_string 
              ("/* declaration of zero size variable " 
               ^ (var_to_string v) 
               ^ " omitted */");
            print_newline ();
            gen_decls count vtl tl s
	   )
       | 32 -> 
	   (
            print_string (typ_ann_opt_to_string (Some t));
            print_space ();
            print_string (var_to_string v);
            print_space ();
            print_string "=";
            print_space ();
            print_string s;
            print_string "[";
            print_int count;
            print_string "];";
            print_newline ();
            gen_decls (count+1) vtl tl s
	   )
       | _ ->  
	   (
            raise (InternalError 
		     "compiler line 877: member field of size other than 32 bits")
	   )
      )
  | _ ->  raise (InternalError 
		   "compiler line 881: wrong number of vars for mutible decl")
;;


(* Preconditions: 
 * The current printer is the desired one.
 * The insertion point is on the begging of the new line.
 *
 * Postconditions: the insertion point will not be on a new line.
 * The same printer will still be current.
 * Any opened block will be closed (the tab stack will have the 
 * same state as when this function was called)
 *)
let gen_mdecl_temp_var (v:var) (ce:cexp):unit =
  let (_,t) = ce in
  (
   print_string (typ_ann_opt_to_string t);
   print_space ();
   print_string (var_to_string v);
   print_space ();
   print_string "=";          
   print_space ();
   gen_cexp ce;
   print_string ";"
  )
;;


(* Purpose: Generate C code for a ? declaration.
 * Seems to handle "let (a,b,c) = ...;"
 * Currently produces (spaces added to keep us in Ocaml comment mode)
 *   Clay_Obj<3> Clay_temp_76 = 
 *        (((Clay_Obj<3> (   *     ) (...))el3_inwStatus)
 *         (ioaddr /  *exp of size zero*  / /  *exp of size zero*  /
 *          /  *exp of size zero*  /));
 *   unsigned long status = Clay_temp_76[0];
 *   unsigned long window = Clay_temp_76[1];
 *   unsigned long busy = Clay_temp_76[2];
 *   /  * declartaion of zero size variable I8599_dev_win_state omitted *  /
 *   /  * declartaion of zero size variable dev_base_state2 omitted *  /
 *   /  * declartaion of zero size variable I8600_dev_busy_state omitted *  /
 *
 * Problem is how to return multiple things and not break memory.
 * C.H.'s solution was to use a templated struct in C++...
 *
 * Can't return an array (returns a reference to a deleted stack frame
 * Don't want to use heap memory
 * Can return a struct
 * Dont want an array in the struct since it makes us fix the array size 
 *   or use heap memory.
 *
 * Possible solution:
 *   typedef struct {
 *     unsigned long tuple_element_0;
 *     unsigned long tuple_element_1;
 *     unsigned long tuple_element_2;
 *   } Clay_tuple_3;
 *   Clay_tuple_3 Clay_temp_76 =
 *        (((Clay_tuple_3 (   *     ) (...))el3_inwStatus)
 *         (ioaddr /  *exp of size zero*  / /  *exp of size zero*  /
 *          /  *exp of size zero*  /));
 *   unsigned long status = Clay_temp_76.tuple_element_0;
 *   unsigned long window = Clay_temp_76.tuple_element_1;
 *   unsigned long busy = Clay_temp_76.tuple_element_2;
 *   /  * declartaion of zero size variable I8599_dev_win_state omitted *  /
 *   /  * declartaion of zero size variable dev_base_state2 omitted *  /
 *   /  * declartaion of zero size variable I8600_dev_busy_state omitted *  /
 *
 * For: returns a copy. generated code is semi-readable. is in C.
 * Against: need a type for each used tuple size.
 * Note: already need a function for each possible return to assemble the tuple.
 *
 * What about the assembly function?
 *
 * Currently: 
 *   inline Clay_Obj<2> Clay_record_contr_37(/* param of size zero*/ 
 *                         unsigned long Clay_record_param_2,
 *                    unsigned long Clay_record_param_3 /*param of size zero*/) {
 *       Clay_Obj<2> record;
 *       /* field of size zero omitted */
 *       record.A[0] = Clay_record_param_2;
 *       record.A[1] = Clay_record_param_3;
 *       /* field of size zero omitted */
 *       return record;
 *   }
 *
 * Possible solution:
 *   inline Clay_tuple_3 Clay_record_contr_37(/* param of size zero*/
 *                         unsigned long Clay_record_param_2,
 *                    unsigned long Clay_record_param_3 /*param of size zero*/) {
 *         Clay_tuple_3 record;
 *       /* field of size zero omitted */
 *       record.tuple_element_0 = Clay_record_param_2;
 *       record.tuple_element_1 = Clay_record_param_3;
 *       /* field of size zero omitted */
 *       return record;
 *   }
 *
 * Note: not sure I really need this many functions. 
 * Can I just have 1 for each used size? 
 * Probly not worth messing with it right now.
 * 
 * Preconditions: 
 * The current printer is the desired one.
 * The insertion point is on the begging of the new line.
 *
 * Postconditions: the insertion point will be on a new line.
 * The same printer will still be current.
 * Any opened block will be closed (the tab stack will have the 
 * same state as when this function was called)
 *)
let gen_mdecl (md:((var ref * typ option) list)*exp):unit = 
  let (l,e) = md in
  match (make_ccode e) with 
  | CExp((ce,t),_) ->
      (
       let v = make_new_temp_var () in
       (
        print_string 
          ( "/* because each mdecl starts a new scope frame,"
            ^" we can declare this each time  */");
        print_newline ();
        gen_mdecl_temp_var v (ce,t);
        print_newline ();
        match t with
        | Some Typ_ann(_,_,Some tlist) -> gen_decls 0 l tlist (var_to_string v)
        | Some Typ_ann(_,_,None) -> 
            raise (InternalError "compiler line 915: missing type annotation")
        | None -> raise (InternalError 
			   "compiler line 917: missing type annotation")
       )
      )
  | CIndStmts(cc_stmts,Some(ce_raw,t)) ->
      (
       let temp = make_new_temp_var () in
       (
        print_string 
          ( "/* because each mdecl starts a new scope frame,"
            ^" we can declare this each time  */");
        print_newline ();
        gen_cstmt_list cc_stmts;
        print_newline ();
        print_string (typ_ann_opt_to_string t);
        print_space ();
        print_string (var_to_string temp);
        print_space ();
        print_string "=";             
        print_space ();
        gen_cexp (ce_raw,t);
        print_string ";";
        print_newline ();
        match t with
        | Some Typ_ann(_,_,Some tlist) -> gen_decls 0 l tlist (var_to_string temp)
        | Some Typ_ann(_,_,None) -> 
            raise (InternalError "compiler line 915: missing type annotation")
        | None -> raise (InternalError 
			   "compiler line 917: missing type annotation")
       )
      )
  | CStmtBlocks(cc_blocks,(ce_raw,t)) ->
      (
       let temp = make_new_temp_var () in
       (
        print_string 
          ( "/* because each mdecl starts a new scope frame,"
            ^" we can declare this each time  */");
        print_newline ();
        gen_cstmt_blocks cc_blocks;
        print_newline ();
        print_string (typ_ann_opt_to_string t);
        print_space ();
        print_string (var_to_string temp);
        print_space ();
        print_string "=";             
        print_space ();
        gen_cexp (ce_raw,t);
        print_string ";";
        print_newline ();
        match t with
        | Some Typ_ann(_,_,Some tlist) -> gen_decls 0 l tlist (var_to_string temp)
        | Some Typ_ann(_,_,None) -> 
            raise (InternalError "compiler line 915: missing type annotation")
        | None -> raise (InternalError 
			   "compiler line 917: missing type annotation")
       )
      )
  | _ -> raise (InternalError 
		  "compiler line 1552: can not assign to an omitted expression")
;;


(*  Assumptions: t has Some of a typ_ann list which has exactly one field 
 *  that is not of size zero, and that this field is exactly one word.
 *)
let get_var_from_one_word_mdecl (md:mdecl):var =
  let (l,e) = md in
  let t = e.exp_typ in
  let (_,index) = 
    (
     match t with 
     | Some Typ_ann(_,_,Some tlist) -> 
         (
          if ((List.length l) = (List.length tlist))
          then find_first_of_size_32
              (List.map 
		 (fun (x:string*typ_ann)->
                   let(s,t) = x in Some(s, Some t)) 
                 tlist
              )
          else raise (InternalError 
			"compiler line 1504: let has a different number of fields than expression being assigned.")
         )
     | Some Typ_ann(_,_,None) -> 
         raise (InternalError "compiler line 938: missing type annotation")
     | None -> 
         raise (InternalError "compiler line 940: missing type annotation")
    )
  in let(v,t) = (List.nth l (index-1)) in !v
;;

(* TODO make sure the {'s and }'s are where they should be *)

(* Purpose: Generate C code for a statement.
 * Statement Block --> { statements }
 * Declaration Statement --> 
 *     size 0 decl:         /* nothing = */  
 *
 * Preconditions: 
 * The current printer is the desired one.
 * The insertion point is at the desired insertion point
 * (the beggining of the line, if desired ).
 *
 * Postconditions: the insertion point will not be on a new line.
 * The same printer will still be current.
 * Any opened block will be closed (the tab stack will have the 
 * same state as when this function was called)
 *
 * Assumptions:  this function assumes that a call to this is not
 * suurounded by curly brackets, and adds them if neccesary.
 *)
let rec gen_stmt (s:stmt) (loop_var_env:code_gen_env):unit =
  let sraw = s.stmt_raw in
  match sraw with 
  | SBlock slist ->
      (
       print_string "{";
       start_block ();
       gen_stmt_list slist loop_var_env;
       end_block ();
       print_string "}"
      )
  | SDecl (uso, d, smt) ->
      (* Declaration Statement: SDecl (need to upack?, ((name, type), expression), statement) *)
      (
       let (({contents = v},_),e) = d in
       let t = e.exp_typ in
       match (size_of_typ_ann_opt t) with
       | 0 -> 
           (
            print_string "{";
            start_block();
            gen_omitted_decl (make_ccode e);
            print_newline ();
            gen_stmt smt loop_var_env;
            end_block ();
            print_string "}"
           )
       | _ ->
           (
            print_string "{";
            start_block();
            gen_decl v (make_ccode e);
            print_newline ();
            gen_stmt smt loop_var_env;
            end_block ();
            print_string "}"
           )
      )
  | SMDecl (uso, md, smt) ->
      (
       let (_,e) = md in
       let t = e.exp_typ in
       match (size_of_typ_ann_opt t) with
       | 0 -> 
           (
            print_string "{";
            start_block();
            gen_omitted_decl (make_ccode e);
            print_newline ();
            gen_stmt smt loop_var_env;
            end_block ();
            print_string "}"
           )
       | 32 -> 
           (
            print_string "{";
            start_block();
            gen_decl (get_var_from_one_word_mdecl md) (make_ccode e);
            print_newline ();
            gen_stmt smt loop_var_env;
            end_block ();
            print_string "}"
           )
       | _ ->
           (
            print_string "{";
            start_block();
            print_string "/* size of expression is not 0 or 1 */";
            print_newline ();
            gen_mdecl md;
            gen_stmt smt loop_var_env;
            end_block ();
            print_string "}"
           )
      )
  | SReturn e -> 
      (
       match (make_ccode e) with
       | CExp(ce,_) ->
	   (
	    print_string "return";
	    print_space ();
	    gen_cexp ce;
	    print_string ";"
	   )
       | CIndStmts(cc_stmts,Some(handle)) ->
	   (
	    print_string "{";
	    start_block ();
	    gen_cstmt_list cc_stmts;
	    print_newline ();
	    print_string "return";
	    print_space ();
	    gen_cexp handle;
	    print_string ";";
	    end_block ();
	    print_string "}"
	   )
       | CIndStmts(cc_stmts,None) ->
	   (
	    print_string "{";
	    start_block ();
	    gen_cstmt_list cc_stmts;
	    print_newline ();
	    print_string "return;";
	    end_block ();
	    print_string "}"
	   )
       | CStmtBlocks(cc_blocks,handle) ->
	   (
	    print_string "{";
	    start_block ();
	    gen_cstmt_blocks cc_blocks;
	    print_newline ();
	    print_string "return";
	    print_space ();
	    gen_cexp handle;
	    print_string ";";
	    end_block ();
	    print_string "}"
	   )
       | Omit -> print_string "return;"
      )
  | SWhile (e, s1, s2) -> 
      (
       match (make_ccode e) with
       | CExp(ce,_) ->
	   (
	    print_string "{";
	    start_block ();
	    let env = make_env [] in
	    (
             print_string "while";
             print_space ();
             print_string "(";
             gen_cexp ce;
             print_string ")";
             print_space();
             gen_stmt s1 env;
             print_newline ()
	    );
	    print_string "/* part of the while loop: do after*/";
	    print_newline ();
	    gen_stmt s2 loop_var_env;
	    end_block ();
	    print_string "}"
	   )
       | CIndStmts(cc_stmts,Some(handle)) ->
	   (
	    print_string "{";
	    start_block ();
	    let env = make_env [] in
	    (
             print_string ("while(" ^ c_true ^ ")");
             print_newline ();
             print_string "{";
             start_block ();
             gen_cstmt_list cc_stmts;
             print_newline ();
             print_string "if(!(";
             gen_cexp handle;
             print_string ")) break;";
             print_newline ();
             gen_stmt s1 env;
             end_block ();
             print_string "}"
	    );
	    print_string "/* part of the while loop: do after*/";
	    print_newline ();
	    gen_stmt s2 loop_var_env;
	    end_block ();
	    print_string "}"
	   )
       | CStmtBlocks(cc_blocks,handle) ->
	   (
	    print_string "{";
	    start_block ();
	    let env = make_env [] in
	    (
             print_string ("while(" ^ c_true ^ ")");
             print_newline ();
             print_string "{";
             start_block ();
             gen_cstmt_blocks cc_blocks;
             print_newline ();
             print_string "if(!(";
             gen_cexp handle;
             print_string ")) break;";
             print_newline ();
             gen_stmt s1 env;
             end_block ();
             print_string "}"
	    );
	    print_string "/* part of the while loop: do after*/";
	    print_newline ();
	    gen_stmt s2 loop_var_env;
	    end_block ();
	    print_string "}"
	   )
       | CIndStmts(cc_stmts,None) -> raise (InternalError "compiler line 2030: can not omit while test.")
       | Omit -> raise (InternalError "compiler line 2030: can not omit while test.")
      )
  | SFor (_,_, loop_decl, test, increment_opt, s1 , s2) ->
      (
       let ((loop_vars:param list), (loop_inits:exp list)) = 
	 List.split loop_decl in
       let env = make_env loop_vars in
       let (pre_block,loop_c_vars) = make_loop_var_decl env loop_inits in
       (
	print_string "{";
	start_block ();
	gen_cstmt_list pre_block;
	gen_loop_var_decl loop_c_vars;
	print_newline ();
	(
	 match (make_ccode test) with
         | CExp(ce,_) ->
             (
              print_string "while";
              print_space ();
              print_string "(";
              gen_cexp ce;
              print_string ")";
              print_space();
              print_newline ();
              print_string "{ /* begin for*/ ";
              start_block ();
             )
         | CIndStmts(cc_stmts, Some(handle)) ->
             (
              print_string ("while(" ^ c_true ^ ")");
              print_newline ();
              print_string "{ /* begin for*/ ";
              start_block ();
              gen_cstmt_list cc_stmts;
              print_newline ();
              print_string "if(!(";
              gen_cexp handle;
              print_string ")) break;";
              print_newline ();
             )
         | CStmtBlocks(cc_blocks,handle) ->
             (
              print_string ("while(" ^ c_true ^ ")");
              print_newline ();
              print_string "{ /* begin for*/ ";
              start_block ();
              gen_cstmt_blocks cc_blocks;
              print_newline ();
              print_string "if(!(";
              gen_cexp handle;
              print_string ")) break;";
              print_newline ();
             )
         | CIndStmts(cc_stmts, None) -> raise (InternalError "compiler line 2272: can not omit if test.")
         | Omit -> raise (InternalError "compiler line 2272: can not omit if test.")
	);
	gen_stmt s1 env;
	(
	 match increment_opt with 
         | Some increment ->
             (
              print_newline ();
              print_string "{ /* begin default loop var. update */";
              start_block ();
              print_string "/* updating values to continue */";
              print_newline ();
              gen_loop_var_update env increment;
              end_block ();
              print_string "} /* end defualt loop var. update */"
             )
         | _ -> ()
	);
	end_block ();
	print_string "} /*end for */";
	print_string "/* part of the for loop: do after*/";
	print_newline ();
	gen_stmt s2 loop_var_env;
	end_block ();
	print_string "}"
       )
      )
  | SContinue elist ->
      (
       print_string "{ /* begin continue loop var. update */";
       start_block ();
       print_string "/* updating values to continue */";
       print_newline ();
       gen_loop_var_update loop_var_env elist;
       print_newline ();
       print_string "continue;";
       end_block ();
       print_string "} /* end continue loop var. update */"
      )
  | SIfElse (e, s1, s2) ->
      (
       match (make_ccode e) with
       | CExp(ce,_) ->
	   (
	    print_string "if";
	    print_space ();
	    print_string "(";
	    gen_cexp ce;
	    print_string ")";
	    print_space ();
	    gen_stmt s1 loop_var_env;
	    print_space ();
	    print_string "else";
	    print_space ();
	    gen_stmt s2 loop_var_env
	   )
       | CIndStmts(cc_stmts,Some(handle)) ->
	   (
	    print_string "{";
	    start_block ();
	    gen_cstmt_list cc_stmts;
	    print_newline ();
	    print_string "if";
	    print_space ();
	    print_string "(";
	    gen_cexp handle;
	    print_string ")";
	    print_space ();
	    gen_stmt s1 loop_var_env;
	    print_space ();
	    print_string "else";
	    print_space ();
	    gen_stmt s2 loop_var_env;
	    end_block ();
	    print_string "}"
	   )
       | CStmtBlocks(cc_blocks,handle) ->
	   (
	    print_string "{";
	    start_block ();
	    gen_cstmt_blocks cc_blocks;
	    print_newline ();
	    print_string "if";
	    print_space ();
	    print_string "(";
	    gen_cexp handle;
	    print_string ")";
	    print_space ();
	    gen_stmt s1 loop_var_env;
	    print_space ();
	    print_string "else";
	    print_space ();
	    gen_stmt s2 loop_var_env;
	    end_block ();
	    print_string "}"
	   )
       | Omit -> raise (InternalError "compiler line 2199: can not omit if test.")
       | CIndStmts(cc_blocks,None) -> raise (InternalError "compiler line 2199: can not omit if test.")
      )
  | SBoolCase _ -> print_string "XXX: BOOL_CASE"
  | SExp e -> 
      (
       match (make_ccode e) with
       | CExp(ce,Must) ->
	   (
	    gen_cexp ce;
	    print_string ";"       (* TODO this doesn't even need to be evaluated if May, does it *)
	   )
       | CExp(ce,May) ->
	   (
	    print_string "/* unnecessary expression omited */"
	   )
       | CIndStmts(cc_stmts,_) ->
	   (
	    print_string "{";
	    start_block ();
	    gen_cstmt_list cc_stmts;
	    end_block ();
	    print_string "}"
	   )
       | CStmtBlocks(cc_blocks,_) ->
	   (
	    print_string "{";
	    start_block ();
	    gen_cstmt_blocks cc_blocks;
	    end_block ();
	    print_string "}"
	   )
       | _ -> print_string "{}"
      )
and gen_stmt_list (slist:stmt list) (loop_var_env:code_gen_env):unit =
  match slist with
  | [] -> ()
  | a::[] -> gen_stmt a loop_var_env
  | a::b -> 
      (
       gen_stmt a loop_var_env; 
       print_newline (); 
       gen_stmt_list b loop_var_env
      )
;;

(*********************** 
 *
 * Function Generation
 *
 ************************)

(* Preconditions: 
 * The current printer is the desired one.
 * The insertion point is on the begging of the new line.
 *
 * Postconditions: the insertion point will be after the last 
 * character that the signature needs (the ")" of the param list).
 * The same printer will still be current.
 *)
let rec gen_fun_sig (f:fun_decl):unit =
  if (f.fun_decl_is_inline = true)
  then 
    (
     print_string "inline";
     print_space_no_indent ()
    );
  let (_, ret_typ_ann) = f.fun_decl_ret in
  print_string (typ_ann_opt_to_string !ret_typ_ann);
  print_space_no_indent ();
  print_string (var_to_string f.fun_decl_name);
  print_break ();
  print_string "(";
  gen_params
    (List.map 
       (fun (x:(param * (typ_ann option ref)))-> 
         let (({contents = v},_),{contents = t})=x in
         if ((size_of_typ_ann_opt t) = 0) then None
         else Some((var_to_string v),t)
       ) f.fun_decl_params
    );
  print_string ")"
;;


(* Preconditions: 
 * The insertion point in the fdefinition_printer and 
 * the fdeclaration_printer is on the begging of the new line.
 *
 * Postconditions: the insertion point will be on a new line in both printers.
 * The same printer that was current prior to this call will be current after.
 * any function prototype will have one line of white space after it; 
 * function definitions will have two.
 *)
let rec gen_fun (f:fun_decl):unit =
  match f with
  | {fun_decl_limit = Limited _} ->  ()
  | {fun_decl_stmt = FunBody s} -> 
      (
       set_printer fprototype_printer;
       (match f.fun_decl_linkage with
       | LinkageCpp -> ()
       | LinkageC -> print_string "extern \"C\" ");
       gen_fun_sig f;
       print_string ";";
       print_newline ();
       print_newline ();
       revert_printer ();
       set_printer fdefinition_printer;
       gen_fun_sig f;
       print_space_no_indent ();
       gen_stmt s (make_env []);
       print_newline ();
       print_newline ();
       print_newline ();
       revert_printer ()
      )
  | {fun_decl_stmt = FunNative} -> 
      (
       set_printer fprototype_printer;
       print_string (match f.fun_decl_linkage with
       | LinkageCpp -> "extern"
       | LinkageC -> "extern \"C\"");
       print_space_no_indent ();
       gen_fun_sig f;
       print_string ";";
       print_newline ();
       print_newline ();
       revert_printer ()
      )
  | {fun_decl_stmt = FunStruct} -> 
      (
       set_printer fprototype_printer;
       let f = (make_struct_constr_decl f) in
       (
	gen_fun_sig f;
	print_string ";";
	print_newline ();
	print_newline ();
	revert_printer ();
	set_printer fdefinition_printer;
	gen_fun_sig f;
	print_space_no_indent ();
	gen_record_or_struct_constr_body
          (
           List.map 
             (fun (x:param * (typ_ann option ref)) ->
               let (({contents = v},_),{contents = t}) = x in
               if ((size_of_typ_ann_opt t) = 0) then None
               else Some((var_to_string v),t)
             ) f.fun_decl_params
          );
	print_newline ();
	print_newline ();
	print_newline ();
	revert_printer ()
       )
      ) 
and make_struct_constr_decl (f:fun_decl):fun_decl =
  {
   fun_decl_is_inline = f.fun_decl_is_inline;
   fun_decl_linkage = f.fun_decl_linkage;
   fun_decl_name = ((get_struct_constructor_name (var_to_string f.fun_decl_name)),0);
   fun_decl_tparams = f.fun_decl_tparams;
   fun_decl_params = f.fun_decl_params;
   fun_decl_ret = f.fun_decl_ret;
   fun_decl_limit = f.fun_decl_limit;
   fun_decl_stmt = f.fun_decl_stmt; 
 }
;; 
 
 let rec gen_funcs (funcs:fun_decl list):unit =
   match funcs with
   | [] -> ()
   | a::b ->
       (
	match a.fun_decl_stmt with
	  FunLocalNative _ -> gen_funcs b  (* since the function is declared natively in the same file, we can skip the extern declaration *)
	| _                -> gen_fun a; 
	    gen_funcs b
       )
;;


(*********************** 
 *
 * C Include Generation
 *
 ************************)

let gen_c_includes = fun (x:unit) ->
  (
   set_printer include_printer;
   print_string "#include \"native.h\"";
   print_newline ();  
   print_newline ()
  )  
;;

(*********************** 
 *
 * CDEmitC handling
 *
 ************************)

let gen_emitted_C (cds:compiler_directive list):unit =
  (
   set_printer emitted_C_printer;
   List.iter
     (fun cd -> 
       match cd with 
	 CDEmit (PureC s) -> print_string s; print_newline () 
       | _ -> ())
     cds
  )
;;


(*********************** 
 *
 * Program Generation
 *
 ************************)

let gen_code (filename:string) (p:program):unit =
  gen_c_includes ();
  gen_funcs(p.program_fun_decls);
  gen_types(p.program_type_decls);
  gen_emitted_C (p.program_directives);
  let cwd = Sys.getcwd () in
  (
   Sys.chdir (Filename.dirname filename);
   let basename = Filename.basename filename in
   let outputfilename = (Filename.chop_extension basename) ^ "_compiled.c" in 
   let ccodelogoutputfilename = (Filename.chop_extension basename) ^ ".log" in (* TODO *)
   (commit_to_file outputfilename; write_log ccode_log_printer ccodelogoutputfilename);
   Sys.chdir cwd
  )
;;

