
open Format;;

(*  Global for defining how far in an indent ot tab should go. *)

let g_indent_spaces = 2;;
let g_tab_spaces = 1;;

type printer = 
  {
   p_formatter:Format.formatter; 
   p_buffer:Buffer.t; 
   p_tab_stack:(int list) ref;
   p_current_tab: int ref;
   p_current_indent: int ref;
   p_indent_spaces: int;
   p_tab_spaces: int;
};;

(*  A small API for switching between printers.  The programmer can naively
 *  just changed printer (i.e. print to different file regions) as needed.
 *  Or, he can call a function that sets it's own printer but reverts before
 *  returning, despite not knowing what the originial printer is.
 *)
let printer_stack = ref [];;

let make_new_printer (b:Buffer.t) (i:int) (s:string):printer = 
  match s with
   | "" -> 
   (
    let f = Format.formatter_of_buffer b in
      (
       Format.pp_set_margin f i;
       Format.pp_open_box f 0;
       {
        p_formatter=f; 
        p_buffer=b; 
        p_tab_stack = ref [];
        p_current_tab = ref 0;
        p_current_indent = ref 0;
        p_indent_spaces = g_indent_spaces;
        p_tab_spaces = g_tab_spaces;
       } 
      )
   )
   | _ -> 
   (
    let f = Format.formatter_of_buffer b in
      (
       Format.pp_set_margin f i;
       Format.pp_open_box f 0;
       Format.pp_print_string f s;
       Format.pp_print_newline f ();
       Format.pp_print_newline f ();
       Format.pp_open_box f 0;
       {
        p_formatter=f; 
        p_buffer=b; 
        p_tab_stack = ref [];
        p_current_tab = ref 0;
        p_current_indent = ref 0;
        p_indent_spaces = g_indent_spaces;
        p_tab_spaces = g_tab_spaces;
       } 
      )
   )
;;

let set_printer (p:printer):unit = printer_stack := p::!printer_stack
;;

let get_printer = fun (x:unit) -> List.hd !printer_stack
;;

let get_formatter = fun (x:unit) -> 
  let p = List.hd !printer_stack in p.p_formatter
;;

let revert_printer = fun (x:unit) -> 
  printer_stack := List.tl !printer_stack
;;


(* Globals for tracking the current state of the printer
 * I.e. where the next line should go.  
 * The difference between an indent and a tab is as follows.
 * The tab is how many spaces in the next line will go.
 * The indent is how many spaces shoudl be added to the tab
 * if the current line spills over
 *
 *)

(*  The print function, its shortcuts, and helpers  
 *
 *  The shortcuts are easiest to use, so I will delineate them here
 * 
 *  print_string s   will cause s to be printed to the current printer
 *
 *  print_int i      will cause i to be printed to the current printer
 *
 *  print_break      tell the current printer that if the next string 
 *                   or int won't fit on this line, then print it on
 *                   the next line at column tab+indent, after setting
 *                   indent to indent_spaces. This has the effect that 
 *                   if a line was a new line and not created by a run 
 *                   over, the next line will be two more collumns to 
 *                   the right.  Otherwise, it will line up.
 *
 *  print_space      tell the current printer that if the next string 
 *                   or int won't fit on this line, then print it on
 *                   the next line at column tab+indent, after setting
 *                   indent to indent_spaces, and if it does fit, print 
 *                   a space.  This has the effect that if a line was 
 *                   a new line and not created by a run over, the next 
 *                   line will be two more collumns to the right.  Other
 *                   wise, it will line up.
 *
 *  print_space_no_indent    tell the current printer that if the next  
 *                   string or int won't fit on this line, then print it
 *                   on the next line at column tab+indent, leaving 
 *                   indent_spaces as it was, and if it does fit, print 
 *                   a space.  This has the effect that if a line is 
 *                   run over, the next line will line up with it.
 *
 *  print_newline    will print a new line and advance the insertion
 *                   point to collumn tab.
 *
 *  print_newline_with_tab t        will print a new line and advance
 *                   the insertion point to collumn t.
 *
 *  start_block      prints a new line, advances the insertion point to
 *                   tab_spaces collumns to the right of the previous
 *                   tab and updates the current tab to this position.  
 *                   To allow end block to work correctly, it pushes the 
 *                   old tab on the stack.  It also sets current indent to 0.
 *  end_block        prints a new line and drops the insertion point to
 *                   the old tab collumn pushed on the stack.  It also
 *                   updates the current tab to this position and sets 
 *                   current indent to 0.
 *
 *   Most of these function figure out what collumn to indent to (if it
 *   makes a new line) by looking at the previous line.  This is based
 *   on the assumption that the current line started at collumn tab + indent
 *)

type break = unit;;
type space = unit;;
type space_no_indent = unit;;
type newline = unit;;
type start_block = unit;;
type end_block = unit;;

type printable = 
  | PString of string
  | PInt of int
  | PBreak
  | PSpace
  | PSpace_no_indent
  | PNewline
  | PNewlinetab of int
  | PStart_block
  | PEnd_block
;;

(* private.  please use functions below *)

let rec print (out:printable):unit =
  match out with
   |  PString s -> (pp_print_string (get_formatter ()) s)
   |  PInt i    -> (pp_print_int (get_formatter ()) i)
   |  PBreak -> 
   (
     let p = get_printer () in
       pp_print_break p.p_formatter 0 0;
       pp_print_if_newline p.p_formatter ();
       pp_print_string p.p_formatter (indent_to_string
        (
	 p.p_current_indent := p.p_indent_spaces;		
         !(p.p_current_tab) + !(p.p_current_indent)
        ))
   )
   |  PSpace -> 
   (
     let p = get_printer () in
       pp_print_break p.p_formatter 1 0;
       pp_print_if_newline p.p_formatter ();
       pp_print_string p.p_formatter (indent_to_string
        (
	 p.p_current_indent := p.p_indent_spaces;		
         !(p.p_current_tab) + !(p.p_current_indent)
        ))
   )
   |  PSpace_no_indent -> 
   (
     let p = get_printer () in
       pp_print_break p.p_formatter 1 0;
       pp_print_if_newline p.p_formatter ();
       pp_print_string p.p_formatter (indent_to_string
        (		
         !(p.p_current_tab) + !(p.p_current_indent)
        ))
   )
   |  PNewline -> 
   (
     let p = get_printer () in
       pp_print_newline p.p_formatter ();     
       pp_open_box p.p_formatter  0;
       p.p_current_indent := 0;
       pp_print_string p.p_formatter (indent_to_string !(p.p_current_tab))
   )
   |  PNewlinetab tab -> 
   (
     let p = get_printer () in
       pp_print_newline p.p_formatter ();  
       pp_open_box p.p_formatter  0;
       p.p_current_indent := 0;
       p.p_current_tab := tab;
       pp_print_string p.p_formatter (indent_to_string !(p.p_current_tab))
   )
   | PStart_block ->
   (
     let p = get_printer () in
       p.p_tab_stack := !(p.p_current_tab) :: !(p.p_tab_stack);
       p.p_current_tab := !(p.p_current_tab) + p.p_tab_spaces;
       pp_print_newline p.p_formatter ();
       pp_open_box p.p_formatter  0;
       p.p_current_indent := 0;
       pp_print_string p.p_formatter (indent_to_string !(p.p_current_tab))
   )
   | PEnd_block ->
   (
     let p = get_printer () in
       (
        match !(p.p_tab_stack) with
           []  -> p.p_current_tab := 0
         | a::b -> 
         (
           p.p_current_tab := a;
           p.p_tab_stack := b;
         );
        (*p.p_current_tab := !(p.p_current_tab) - p.p_tab_spaces;*)
        p.p_current_indent := 0;
        pp_print_newline p.p_formatter (); 
        pp_print_string p.p_formatter (indent_to_string !(p.p_current_tab))
       )
   )
and indent_to_string (i:int):string =
  if i=0 then "" else (" " ^ (indent_to_string (i-1)))
;;

(* These are the public printing functions.  Please only use these *)

let print_string = fun (s:string) -> print (PString s);;
let print_int = fun (i:int) -> print (PInt i);;
let print_break = fun (x:unit)-> print PBreak;;
let print_space = fun (x:unit) -> print PSpace;;
let print_space_no_indent = fun (x:unit) -> print PSpace_no_indent;;
let print_newline = fun (x:unit) -> print PNewline;;
let print_newline_with_tab = fun (i:int) -> print (PNewlinetab i);;
let start_block = fun (x:unit) -> print PStart_block;;
let end_block = fun (x:unit) -> print PEnd_block;;
let printf = fun (s:('a, unit, string) format) -> 
  print (PString (sprintf s));; 

