%{
open Ast;;
open Big_int;;
open Arith;;

let exp_here (exp_raw:exp_raw) = {exp_raw = exp_raw; exp_pos = Some (Parsing.symbol_start ()); exp_typ = None};;
let stmt_here (stmt_raw:stmt_raw) = {stmt_raw = stmt_raw; stmt_pos = Some (Parsing.symbol_start ())};;

(* let rec parse_error s = raise (PosExn (Parsing.symbol_start (), SyntaxError "syntax error"));; *)
let rec parse_error s = raise (SyntaxError "");;

(*
  MultOp | DivOp | ModOp
| LShiftOp | RShiftOp
| BitwiseAndOp | XorOp | BitwiseOrOp
*)

let t2i (t:typ):int_arith =
  match t with
  | TVar x -> iarith_var x
  | TInt t -> t
  | _ -> parse_error ""
;;

let t2b (t:typ):bool_arith =
  match t with
  | TVar x -> BVar x
  | TBool t -> t
  | _ -> parse_error ""
;;

(* ebinary takes a bin_op and 2 expressions and returns an expression.
   I think the return exp is prepared to call the op on the exp arguments.
*)
let ebinary (op:binary_op) (e1:exp) (e2:exp):exp =
  let names =
    match op with
    | AddOp -> ["i32_add"; "s32_add"; "u32_add"]
    | SubOp -> ["i32_subtract"; "s32_subtract"; "u32_subtract"]
    | MultOp -> ["s32_mult"]
    | LtOp -> ["is32_lt"; "iu32_lt"; "s32_lt"; "u32_lt"]
    | GtOp -> ["is32_gt"; "iu32_gt"; "s32_gt"; "u32_gt"]
    | LeOp -> ["is32_le"; "iu32_le"; "s32_le"; "u32_le"]
    | GeOp -> ["is32_ge"; "iu32_ge"; "s32_ge"; "u32_ge"]
    | EqOp -> ["is32_eq"; "iu32_eq"; "s32_eq"; "u32_eq"]
    | NeOp -> ["is32_ne"; "iu32_ne"; "s32_ne"; "u32_ne"]
    | AndOp -> ["bool_and"]
    | OrOp -> ["bool_or"]
    | LShiftOp -> ["iu32_lshift"; "s32_lshift"]
    | RShiftOp -> ["iu8_rshift"; "iu16_rshift"; "iu32_rshift"; "s32_rshift"]
    | BitwiseAndOp -> ["iu32_and"; "s32_and"; "u32_and"]
    | BitwiseOrOp -> ["iu32_or"; "s32_or"; "u32_or"]
    | BitwiseXorOp -> ["iu32_xor"; "s32_xor"; "u32_xor"]
    | _ -> raise (NotImplemented "parse.mly: ebinary")
  in exp_here (ECall (exp_here (EOverload (names, ref None)), [e1;e2]))
;;

(* eunary takes a unary_op and one expression and returns an expression.
   I think the return exp is prepared to call the op on the exp argument.
*)
let eunary (op:unary_op) (e:exp):exp =
  let names =
    match op with
    | NotOp -> ["bool_not"]
    | BitwiseNotOp -> ["is32_not"]
    | _ -> raise (NotImplemented "parse.mly: eunary")
  in exp_here (ECall (exp_here (EOverload (names, ref None)), [e]))
;;

(* process_statement_list takes a stmt list, 
   may process one stmt and returns another stmt list.
*)
let rec process_statement_list (slist:stmt list):stmt list =
  match slist with
  | [] -> []
  | {stmt_raw = SDecl (u, d, _); stmt_pos = Some p}::t -> [stmt_at (SDecl (u, d, stmt_at (SBlock (process_statement_list t)) p)) p]
  | {stmt_raw = SMDecl (u, d, _); stmt_pos = Some p}::t -> [stmt_at (SMDecl (u, d, stmt_at (SBlock (process_statement_list t)) p)) p]
  | {stmt_raw = SWhile (e, s2, _); stmt_pos = Some p}::t -> [stmt_at (SWhile (e, s2, stmt_at (SBlock (process_statement_list t)) p)) p]
  | {stmt_raw = SFor (tp, w, ps, e, a, s2, _); stmt_pos = Some p}::t -> [stmt_at (SFor (tp, w, ps, e, a, s2, stmt_at (SBlock (process_statement_list t)) p)) p]
  | h::t -> h::(process_statement_list t)
;;

let rec conjuncts_to_bool (conjuncts:bool_arith list):bool_arith =
  match conjuncts with
  | [] -> BConst true
  | [h] -> h
  | h::t -> BBinary (BAndOp, h, conjuncts_to_bool t)
;;

%}

%token AUTO
%token BREAK
%token CASE
%token CHAR
%token CONST
%token CONTINUE
%token DEFAULT
%token DO
%token DOUBLE
%token ELSE
%token ENUM
%token EXTERN
%token FLOAT
%token FOR
%token GOTO
%token IF
%token INT
%token INLINE
%token LONG
%token REGISTER
%token RETURN
%token SHORT
%token SIGNED
%token SIZEOF
%token STATIC
%token STRUCT
%token SWITCH
%token TYPEDEF
%token UNION
%token UNSIGNED
%token VOID
%token VOLATILE
%token WHILE

%token BOOL
%token NEW
%token NULL
%token TRY
%token CATCH
/*
%token READONLY
%token SUBARRAY
%token MODULE
%token INTERFACE
*/
%token IMPORT
%token EXPORTS
%token ALL
/*
%token THREAD
%token SPAWN
%token ASYNC
*/
%token LET
%token WHERE
%token <int> TYPE
%token TBOOL
%token <Ast.sign * int> SIZEDTINT
%token <Ast.sign * int> SIZEDINT
%token BYTE
%token NATIVE
%token <string> NATIVEBLOCK
%token ALL
%token EXISTS
%token FUN
%token TYPEFUN
%token REC
%token PACK
%token UNPACK
%token EQUIV
%token LIMITED
%token LIMITANY

%token LBRACKET
%token RBRACKET
%token LPAREN
%token RPAREN
%token DOT
%token LARROW
%token RARROW
%token PLUSPLUS
%token MINUSMINUS
%token AMPERSAND
%token STAR
%token PLUS
%token MINUS
%token TILDE
%token BANG
%token SLASH
%token PERCENT
%token LTLT
%token GTGT
%token LT
%token GT
%token LE
%token GE
%token EQUALSEQUALS
%token NE
%token CARET
%token BAR
%token AMPERSANDAMPERSAND
%token BARBAR
%token QUESTION
%token COLON
%token EQUALS
%token STAREQUALS
%token SLASHEQUALS
%token PERCENTEQUALS
%token PLUSEQUALS
%token MINUSEQUALS
%token LTLTEQUALS
%token GTGTEQUALS
%token AMPERSANDEQUALS
%token CARETEQUALS
%token BAREQUALS
%token COMMA
%token LBRACE
%token RBRACE
%token SEMICOLON
%token AT

%token HEAP

%token <string> AQEXPR
%token <string> AQTYPE
%token <string> AQSTMT

%token <string> ID
%token <string> UID
%token <string> STRING_LITERAL
%token <Big_int.big_int> CONSTINT
%token <bool> CONSTBOOL
%token EOF
%token BAD_TOKEN





%type <Ast.program> program


%start program

%%

program:
  top_decls EOF { $1 }
;

top_decls:
  top_decl              { $1 }
| top_decl top_decls
  {
    { program_type_decls = $1.program_type_decls @ $2.program_type_decls;
      program_fun_decls = $1.program_fun_decls @ $2.program_fun_decls;
      program_directives = $1.program_directives @ $2.program_directives
    }
  }
;

top_decl:
  compiler_directive 
  { { program_type_decls = []; program_fun_decls = []; program_directives = [$1] } }
| type_definition 
  { { program_type_decls = [$1]; program_fun_decls = []; program_directives = [] } }
| function_definition 
  { let fcn = $1 in
    match fcn.fun_decl_stmt with
	FunLocalNative s -> { program_type_decls = []; program_fun_decls = [fcn]; 
				program_directives = [CDEmit (PureC s)] }
	| _              -> { program_type_decls = []; program_fun_decls = [fcn]; 
				program_directives = []; }
  }
;

compiler_directive:
  IMPORT symbol_list SEMICOLON    { CDImport $2 }
| EXPORTS symbol_list SEMICOLON   { CDExports $2 }
| EXPORTS ALL SEMICOLON           { CDExportsAll }
| NATIVEBLOCK                     { CDEmit (PureC $1)  }
;

symbol_list:
  var                      { [CDSymbol (fst $1)] }
| tvar                     { [CDSymbol (fst $1)] }
| STRING_LITERAL           { [CDLiteral $1] }
| var COMMA symbol_list    { (CDSymbol (fst $1)) :: $3 }
| tvar COMMA symbol_list   { (CDSymbol (fst $1)) :: $3 }
| STRING_LITERAL COMMA symbol_list { (CDLiteral $1) :: $3 }
;

function_definition:
  etyp var type_parameters_where_opt LPAREN parameter_list_opt RPAREN limit_opt 
	compound_statement
  {
    { fun_decl_is_inline = false;
      fun_decl_linkage = LinkageCpp;
      fun_decl_name = $2;
      fun_decl_tparams = $3;
      fun_decl_params = $5;
      fun_decl_ret = ($1, ref None);
      fun_decl_limit = $7;
      fun_decl_stmt = FunBody $8;
    }
  }
| linkage_spec_c etyp var type_parameters_where_opt LPAREN parameter_list_opt RPAREN 
	limit_opt compound_statement
  {
    { fun_decl_is_inline = false;
      fun_decl_linkage = $1;
      fun_decl_name = $3;	
      fun_decl_tparams = $4;
      fun_decl_params = $6;
      fun_decl_ret = ($2, ref None);
      fun_decl_limit = $8;
      fun_decl_stmt = FunBody $9;
    }
  }
| linkage_spec INLINE etyp var type_parameters_where_opt LPAREN parameter_list_opt 
	RPAREN limit_opt compound_statement
  {
    { fun_decl_is_inline = true;
      fun_decl_linkage = $1;
      fun_decl_name = $4;	
      fun_decl_tparams = $5;
      fun_decl_params = $7;
      fun_decl_ret = ($3, ref None);
      fun_decl_limit = $9;
      fun_decl_stmt = FunBody $10;
    }
  }
| NATIVE linkage_spec etyp var type_parameters_where_opt LPAREN parameter_list_opt 
	RPAREN limit_opt SEMICOLON
  {
    { fun_decl_is_inline = false;
      fun_decl_linkage = $2;
      fun_decl_name = $4;
      fun_decl_tparams = $5;
      fun_decl_params = $7;
      fun_decl_ret = ($3, ref None);
      fun_decl_limit = $9;
      fun_decl_stmt = FunNative;
    }
  }
| NATIVE linkage_spec INLINE etyp var type_parameters_where_opt LPAREN parameter_list_opt 
	RPAREN limit_opt SEMICOLON
  {
    { fun_decl_is_inline = true;
      fun_decl_linkage = $2;
      fun_decl_name = $5;
      fun_decl_tparams = $6;
      fun_decl_params = $8;
      fun_decl_ret = ($4, ref None);
      fun_decl_limit = $10;
      fun_decl_stmt = FunNative;
    }
  }
| etyp var type_parameters_where_opt LPAREN parameter_list_opt RPAREN limit_opt EQUALS 
	NATIVEBLOCK
  {
    { fun_decl_is_inline = false;
      fun_decl_linkage = LinkageCpp;
      fun_decl_name = $2;
      fun_decl_tparams = $3;
      fun_decl_params = $5;
      fun_decl_ret = ($1, ref None);
      fun_decl_limit = $7;
      fun_decl_stmt = FunLocalNative $9;
    }
  }
| linkage_spec_c etyp var type_parameters_where_opt LPAREN parameter_list_opt RPAREN 
	limit_opt EQUALS NATIVEBLOCK
  {
    { fun_decl_is_inline = false;
      fun_decl_linkage = $1;
      fun_decl_name = $3;
      fun_decl_tparams = $4;
      fun_decl_params = $6;
      fun_decl_ret = ($2, ref None);
      fun_decl_limit = $8;
      fun_decl_stmt = FunLocalNative $10;
    }
  }
;

linkage_spec:
  { LinkageCpp }
| linkage_spec_c { $1 }
;

linkage_spec_c:
  EXTERN STRING_LITERAL { if $2 = "\"C\"" then LinkageC else raise (SyntaxError "unknown linkage specifier") }
;

limit_opt:
  { Unlimited }
| LIMITANY { LimitAny }
| LIMITED LBRACKET typ RBRACKET { Limited (t2i $3) }
;

parameter_list_opt: { [] } | parameter_list { $1 };
parameter_list:
  parameter_declaration	{ [$1] }
| parameter_declaration COMMA parameter_list { $1::$3 }
;

parameter_declaration:
  etyp var { ((ref $2, $1), ref None) }
;

decl_list_opt: { [] } | decl_list { $1 } ;
decl_list:
  decl	{ [$1] }
| decl COMMA decl_list { $1::$3 }
;

decl:
  var { (ref $1, None) }
| etyp var { (ref $2, Some $1) }
;

linear_opt:
     {Nonlinear}
| AT {Linear}
;

type_definition:
  TYPEDEF tvar EQUALS typ { ($2, AbbrevSpec $4) }
| TYPEDEF tvar type_parameters EQUALS typ { ($2, AbbrevSpec (TFun ($3, $5))) }
| kind2 tvar type_parameters_opt EQUALS struct_spec
  {
    match $1 with
    | KType (size, lin) -> ($2, StructSpec (size, lin, $3, $5))
    | _ -> parse_error ""
  }
| kind2 tvar type_parameters_opt EQUALS NATIVE
  {
    match $1 with
    | KType (size, lin) -> ($2, NativeSpec (size, lin, $3))
    | _ -> parse_error ""
  }
| ENUM tvar LBRACE enum_spec RBRACE { ($2, EnumSpec $4) }
/* | HEAP tvar type_parameters_opt LBRACE field_list RBRACE { ($2, HeapSpec ($3, $5)) } */
;

type_parameters:
  LBRACKET type_parameter_list RBRACKET { $2 }
;
type_parameters_opt:
  { None }
| LBRACKET type_parameter_list RBRACKET { Some $2 }
;
type_parameters_where:
  LBRACKET type_parameter_where_list where_opt RBRACKET { (fst $2, conjuncts_to_bool ((snd $2) @ $3)) }
;

type_parameters_where_opt:
  { None }
| LBRACKET type_parameter_where_list where_opt RBRACKET { Some (fst $2, conjuncts_to_bool ((snd $2) @ $3)) }
;
type_parameter_list: { [] } | type_parameter_list_1 { $1 };
type_parameter_list_1:
  kind tvar { [($2, $1)] }
| kind tvar COMMA type_parameter_list_1 { ($2, $1)::$4 }
;

type_parameter_where_list: { ([],[]) } | type_parameter_where_list_1 { $1 };
type_parameter_where_list_1:
  kind_or_int_tvar { let (tparam, bools) = $1 in ([tparam], bools) }
| kind_or_int_tvar COMMA type_parameter_where_list_1 { let (tparam, bools) = $1 in (tparam::(fst $3), bools @ (snd $3)) }
;
kind_or_int_tvar:
  kind tvar { (($2, $1), []) }
| SIZEDINT tvar
  {
    let (sign, nbits) = $1 in
    let size = Big_int.power_int_positive_int 2 nbits  in
    let max_n = (match sign with
      | Signed -> Big_int.div_big_int size (Big_int.big_int_of_int 2)
      | Unsigned -> size) in
    let min_n = Big_int.sub_big_int max_n size in
    (($2, KInt), [BBinary
      ( BAndOp,
        BCompare (BLeOp, iarith_const min_n, iarith_var $2),
        BCompare (BLtOp, iarith_var $2, iarith_const max_n))])
  }
;

where_opt:
  { [] }
| SEMICOLON bool_arith { [$2] }
;

struct_spec:
  STRUCT LBRACE field_list RBRACE  { $3 }
;

field_list:
  { [] }
| etyp field_id SEMICOLON field_list { ($2, $1)::$4 };
;

enum_spec:
  var { [($1, None)] }
| var COMMA enum_spec { (($1, None) :: $3) }
| var EQUALS CONSTINT COMMA enum_spec { (($1, Some $3) :: $5) }
;

/*
The kinds (minus precedence and associativity) are
  INT | BOOL				int X, bool Y
  linear_opt TYPE			@type0  @type  type
  kind LARROW LPAREN kind_list RPAREN	@type0<-(int)
*/
kind:
  kind2 { K2 $1 }
| INT { KInt }
| BOOL { KBool }
;

kind2:
  linear_opt TYPE { KType ($2, $1) }
/*| LPAREN kind_list RPAREN ARROW kind2 { KArrow ($2, $5) }*/
| kind2 LARROW LPAREN kind_list RPAREN { KArrow ($4, $1) }
;

kind_list: { [] } | kind_list_1 { $1 };
kind_list_1:
  kind                   { [$1] }
| kind COMMA kind_list_1 { $1::$3 }
;

/*
int_arith: int_arith1 { $1 };

int_arith1:
  int_arith2 { $1 }
| int_arith1 PLUS int_arith2 { add_int_arith $1 $3 }
| int_arith1 MINUS int_arith2 { add_int_arith $1 (mult_int_arith (big_int_of_int (-1)) $3) }
;

int_arith2:
  LPAREN int_arith RPAREN     { $2 }
| CONSTINT STAR int_arith2   { mult_int_arith $1 $3 }
| CONSTINT        { IConst $1 }
| tvar            { IVar $1 }
;

typ_int_arith:
| int_arith1 PLUS int_arith2 { add_int_arith $1 $3 }
| int_arith1 MINUS int_arith2 { add_int_arith $1 (mult_int_arith (big_int_of_int (-1)) $3) }
| CONSTINT STAR int_arith2   { mult_int_arith $1 $3 }
| CONSTINT        { IConst $1 }
;

bool_arith: bool_arith1 { $1 };

bool_arith1:
  bool_arith2 { $1 }
| bool_arith1 BARBAR bool_arith2 { BBinary (BOrOp, $1, $3) }
;

bool_arith2:
  bool_arith3 { $1 }
| bool_arith2 AMPERSANDAMPERSAND bool_arith3 { BBinary (BAndOp, $1, $3) }
;

bool_arith3:
  bool_arith4 { $1 }
| int_arith EQUALSEQUALS int_arith { BCompare (BEqOp, t2i $1, t2i $3) }
| int_arith NE int_arith { BCompare (BNeOp, t2i $1, t2i $3) }
| int_arith GE int_arith { BCompare (BGeOp, t2i $1, t2i $3) }
| int_arith LE int_arith { BCompare (BLeOp, t2i $1, t2i $3) }
| int_arith GT int_arith { BCompare (BGtOp, t2i $1, t2i $3) }
| int_arith LT int_arith { BCompare (BLtOp, t2i $1, t2i $3) }
| BANG bool_arith4 { BNot $2 }
;

bool_arith4:
  LPAREN bool_arith RPAREN  { $2 }
| CONSTBOOL       { BConst $1 }
| tvar            { BVar $1 }
;

typ_bool_arith:
| bool_arith1 BARBAR bool_arith2 { BBinary (BOrOp, $1, $3) }
| bool_arith2 AMPERSANDAMPERSAND bool_arith3 { BBinary (BAndOp, $1, $3) }
| int_arith EQUALSEQUALS int_arith { BCompare (BEqOp, $1, $3) }
| int_arith NE int_arith { BCompare (BNeOp, $1, $3) }
| int_arith GE int_arith { BCompare (BGeOp, $1, $3) }
| int_arith LE int_arith { BCompare (BLeOp, $1, $3) }
| int_arith GT int_arith { BCompare (BGtOp, $1, $3) }
| int_arith LT int_arith { BCompare (BLtOp, $1, $3) }
| BANG bool_arith4 { BNot $2 }
| CONSTBOOL       { BConst $1 }
;
*/

bool_arith: typ3 { t2b $1 }
/*int_arith: typ7 { t2i $1 }*/

/*
etyp (minus precedence)
  etyp LARROW limit_opt LPAREN type_list RPAREN 
  ALL type_parameters_where etyp		all [int X] Int[X]
  EXISTS type_parameters_where etyp		exists [int X] Int[X]
  etyp LBRACKET type_list RBRACKET		Mem[A,V]
  LPAREN etyp RPAREN				???
  INT | BYTE | BOOL | VOID			int, byte, bool, void
  tvar						X
  DOT LBRACKET rec_fields RBRACKET		.[]
  AT LBRACKET rec_fields RBRACKET		@[]
  AQTYPE
*/
etyp: etyp1 { $1 };

etyp1:
  etyp2  { $1 }
/*XXX: what to do about the TArrow syntax?  The following is ambiguous */
/*| LPAREN type_list RPAREN RARROW typ1 { TArrow ($2, $5) }*/
/* The following works, but do we want to use "fun" for two different things? */
/*| FUN LPAREN type_list RPAREN RARROW etyp1 { TArrow ($3, $6) }*/
/* The following works, but it's sure ugly! */
| etyp2 LARROW limit_opt LPAREN type_list RPAREN { TArrow ($5, $1, $3) }
| ALL type_parameters_where etyp1 { TAll (fst $2, snd $2, $3) }
| EXISTS type_parameters_where etyp1 { TExists (fst $2, snd $2, $3) }
;

etyp2:
  etyp3  { $1 }
| etyp2 LBRACKET type_list RBRACKET { TApp ($1, $3) }
;

etyp3:
  LPAREN etyp RPAREN  { $2 }
| INT   { TVar ("__Int", 0) }
| BYTE  { TVar ("__Byte", 0) }
| BOOL  { TVar ("__Bool", 0) }
| VOID  { TVar ("__Unit", 0) }
| tvar            { TVar $1 }
| DOT LBRACKET rec_fields RBRACKET { TRecord (Nonlinear, $3) }
| AT LBRACKET rec_fields RBRACKET { TRecord (Linear, $3) }
| AQTYPE          { TAntiquote $1 }
;

/*
typ (minus precedence and associativity)
  typ LARROW limit_opt LPAREN type_list RPAREN
  ALL type_parameters_where typ
  EXISTS type_parameters_where typ
  FUN type_parameters typ
  typ LBRACKET type_list RBRACKET
  typ ( BARBAR | AMPERSANDAMPERSAND ) typ
  typ ( EQUALSEQUALS | NE | GE | LE | GT | LT) typ
  type BANG typ
  typ ( PLUS | MINUS ) typ
  MINUS typ
  CONSTINT | CONSTBOOL
  CONSTINT STAR typ | type STAR CONSTINT 	3 * X		X * 3
  LPAREN typ RPAREN
  INT | BYTE | BOOL | VOID
  tvar
  DOT LBRACKET rec_fields RBRACKET
  AT LBRACKET rec_fields RBRACKET
  AQTYPE
*/
typ: typ1 { $1 };

typ1:
  typ2  { $1 }
/*| LPAREN type_list RPAREN RARROW typ1 { TArrow ($2, $5) }*/
/*| FUN LPAREN type_list RPAREN RARROW typ1 { TArrow ($3, $6) }*/
| typ2 LARROW limit_opt LPAREN type_list RPAREN { TArrow ($5, $1, $3) }
| ALL type_parameters_where typ1 { TAll (fst $2, snd $2, $3) }
| EXISTS type_parameters_where typ1 { TExists (fst $2, snd $2, $3) }
| FUN type_parameters typ1 { TFun ($2, $3) }
;

typ2:
  typ3  { $1 }
| typ2 LBRACKET type_list RBRACKET { TApp ($1, $3) }
;

typ3:
  typ4 { $1 }
| typ3 BARBAR typ4 { TBool (BBinary (BOrOp, t2b $1, t2b $3)) }
;

typ4:
  typ5 { $1 }
| typ4 AMPERSANDAMPERSAND typ5 { TBool (BBinary (BAndOp, t2b $1, t2b $3)) }
;

typ5:
  typ6 { $1 }
| typ6 EQUALSEQUALS typ6 { TBool (BCompare (BEqOp, t2i $1, t2i $3)) }
| typ6 NE typ6 { TBool (BCompare (BNeOp, t2i $1, t2i $3)) }
| typ6 GE typ6 { TBool (BCompare (BGeOp, t2i $1, t2i $3)) }
| typ6 LE typ6 { TBool (BCompare (BLeOp, t2i $1, t2i $3)) }
| typ6 GT typ6 { TBool (BCompare (BGtOp, t2i $1, t2i $3)) }
| typ6 LT typ6 { TBool (BCompare (BLtOp, t2i $1, t2i $3)) }
;

typ6:
  typ7 { $1 }
| BANG typ6 { TBool (BNot (t2b $2)) }
;

typ7:
  typ8 { $1 }
| typ7 PLUS typ8 { TInt (add_int_arith (t2i $1) (t2i $3)) }
| typ7 MINUS typ8 { TInt (add_int_arith (t2i $1) (mult_int_arith (big_int_of_int (-1)) (t2i $3))) }
| MINUS typ8 { TInt (mult_int_arith (big_int_of_int (-1)) (t2i $2)) }
;

typ8:
  typ9 { $1 }
| CONSTINT        { TInt (iarith_const $1) }
| CONSTINT STAR typ8 { TInt (mult_int_arith $1 (t2i $3)) }
;

typ9:
  LPAREN typ RPAREN  { $2 }
| INT   { TVar ("__Int", 0) }
| BYTE  { TVar ("__Byte", 0) }
| BOOL  { TVar ("__Bool", 0) }
| VOID  { TVar ("__Unit", 0) }
| tvar            { TVar $1 }
| CONSTBOOL       { TBool (BConst $1) }
| typ9 STAR CONSTINT { TInt (mult_int_arith $3 (t2i $1)) }
| DOT LBRACKET rec_fields RBRACKET { TRecord (Nonlinear, $3) }
| AT LBRACKET rec_fields RBRACKET { TRecord (Linear, $3) }
| AQTYPE          { TAntiquote $1 }
;

type_list_opt: { [] } | type_list { $1 };
type_list:
  typ                   { [$1] }
| typ COMMA type_list { $1::$3 }
;

rec_fields: { [] } | rec_fields_1 { $1 };
rec_fields_1:
  typ { [("1", $1)] }
| typ field_id { [($2, $1)] }
| rec_fields_1 COMMA typ { $1 @ [((string_of_int (1 + (List.length $1))), $3)] }
| rec_fields_1 COMMA typ field_id { $1 @ [($4, $3)] }
;

statement:
  compound_statement    { $1 }
| expression_statement  { $1 }
| selection_statement   { $1 }
| iteration_statement   { $1 }
| for_statement         { $1 }
| jump_statement        { $1 }
| decl_statement        { $1 }
| AQSTMT                { stmt_here (SAntiquote $1) }
;

e_statement:
  compound_statement    { $1 }
| expression_statement  { $1 }
| e_selection_statement { $1 }
| e_iteration_statement { $1 }
| for_statement         { $1 }
| jump_statement        { $1 }
| AQSTMT                { stmt_here (SAntiquote $1) }
;

jump_statement:
  RETURN expression SEMICOLON   { stmt_here (SReturn $2) }
| RETURN SEMICOLON              { stmt_here (SReturn (exp_here EUnit)) }
| CONTINUE SEMICOLON            { stmt_here (SContinue []) }
| CONTINUE LPAREN argument_expression_list RPAREN SEMICOLON { stmt_here (SContinue $3) }
;

compound_statement:
  LBRACE statement_list RBRACE  { stmt_here (SBlock $2) }
;

statement_list: { [] } | statement_list_1 { process_statement_list $1 };
statement_list_1:
  statement                   { [$1] }
| statement statement_list_1  { $1::$2 }
;

expression_statement:
  SEMICOLON             { stmt_here (SExp (exp_here EUnit)) }
| expression SEMICOLON  { stmt_here (SExp $1) }
;

iteration_statement:
   WHILE LPAREN expression RPAREN statement  { stmt_here (SWhile ($3, $5, stmt_here (SBlock []))) }
| FOR LPAREN statement expression SEMICOLON expression RPAREN statement 
  { 
    stmt_here (SBlock (process_statement_list [$3;(stmt_here 
        (SWhile ($4, stmt_here (SBlock (process_statement_list 
            [$8;stmt_here (SExp $6)])), stmt_here (SBlock []))))])) 
  }
;

e_iteration_statement:
  WHILE LPAREN expression RPAREN e_statement  { stmt_here (SWhile ($3, $5, stmt_here (SBlock []))) }
;

for_statement:
  FOR type_parameters_where LPAREN for_decl_list SEMICOLON expression RPAREN for_iter_opt compound_statement
  {
    stmt_here (SFor (fst $2, snd $2, $4, $6, $8, $9, stmt_here (SBlock [])))
  }
;
for_iter_opt:
  { None }
| LPAREN argument_expression_list RPAREN { Some $2 }
;

for_decl:
  typ var EQUALS expression { ((ref $2, $1), $4) }
;
for_decl_list:
  { [] }
| for_decl_list1 { $1 }
;
for_decl_list1:
  for_decl { [$1] }
| for_decl COMMA for_decl_list1 { $1::$3 }
;

selection_statement:
  IF LPAREN expression RPAREN statement                     { stmt_here (SIfElse ($3, $5, stmt_here (SExp (exp_here EUnit)))) }
| IF LPAREN expression RPAREN e_statement ELSE statement    { stmt_here (SIfElse ($3, $5, $7)) }
| IF LBRACKET typ RBRACKET statement                     { stmt_here (SBoolCase (t2b $3, $5, stmt_here (SExp (exp_here EUnit)))) }
| IF LBRACKET typ RBRACKET e_statement ELSE statement    { stmt_here (SBoolCase (t2b $3, $5, $7)) }
;

e_selection_statement:
  IF LPAREN expression RPAREN e_statement ELSE e_statement  { stmt_here (SIfElse ($3, $5, $7)) }
| IF LBRACKET typ RBRACKET e_statement ELSE e_statement  { stmt_here (SBoolCase (t2b $3, $5, $7)) }
;

decl_statement:
  etyp var EQUALS expression SEMICOLON   { stmt_here (SDecl (None, ((ref $2, Some $1), $4), stmt_here (SBlock []))) }
| LET unpack_opt etyp var EQUALS expression SEMICOLON   { stmt_here (SDecl ($2, ((ref $4, Some $3), $6), stmt_here (SBlock []))) }
| LET unpack_opt var EQUALS expression SEMICOLON   { stmt_here (SDecl ($2, ((ref $3, None), $5), stmt_here (SBlock []))) }
| LET unpack_opt LPAREN decl_list_opt RPAREN EQUALS expression SEMICOLON   { stmt_here (SMDecl ($2, ($4, $7), stmt_here (SBlock []))) }
/*
| UNPACK LBRACKET unpack_list RBRACKET etyp var EQUALS expression SEMICOLON   { stmt_here (SDecl (Some $3, (($6, Some $5), $8), stmt_here (SBlock []))) }
| UNPACK LBRACKET unpack_list RBRACKET var EQUALS expression SEMICOLON   { stmt_here (SDecl (Some $3, (($5, None), $7), stmt_here (SBlock []))) }
| UNPACK LBRACKET unpack_list RBRACKET LPAREN decl_list RPAREN EQUALS expression SEMICOLON   { stmt_here (SMDecl (Some $3, ($6, $9), stmt_here (SBlock []))) }
*/
;

unpack_opt:
  { None }
| LBRACKET unpack_list RBRACKET { Some $2 }
;

primary_expression:
  var         { exp_here (EVar (ref $1)) }
| STRUCT tvar  { exp_here (EStruct $2) }
/* | NULL */
| CONSTINT    { exp_here (EInt $1) }
| CONSTBOOL   { exp_here (EBool $1) }
/* | STRING_LITERAL */
| LPAREN expression RPAREN  { $2 }
| DOT LPAREN rec_init_list RPAREN { exp_here (ERecord (Nonlinear, $3)) }
| AT LPAREN rec_init_list RPAREN { exp_here (ERecord (Linear, $3)) }
| PACK LBRACKET typ RBRACKET LBRACKET type_app_list RBRACKET LPAREN expression RPAREN { exp_here (EPack ($9, $3, $6)) }
| PACK LBRACKET typ RBRACKET LPAREN expression RPAREN { exp_here (EPack ($6, $3, [])) }
| AQEXPR      { exp_here (EAntiquote $1) }
;

rec_init_list: { [] } | rec_init_list_1 { $1 };
rec_init_list_1:
  conditional_expression { [("1", None, $1)] }
| field_id EQUALS conditional_expression { [($1, None, $3)] }
| etyp field_id EQUALS conditional_expression { [($2, Some $1, $4)] }
| rec_init_list_1 COMMA conditional_expression { $1 @ [(string_of_int (1 + (List.length $1)), None, $3)] }
| rec_init_list_1 COMMA field_id EQUALS conditional_expression { $1 @ [($3, None, $5)] }
| rec_init_list_1 COMMA etyp field_id EQUALS conditional_expression { $1 @ [($4, Some $3, $6)] }
;

expression:
  assignment_expression                   { $1 }
/* | expression COMMA assignment_expression  */
;

/*
constant_expression:
  conditional_expression    { $1 }
;
*/

conditional_expression:
  logical_OR_expression     { $1 }
/*| logical_OR_expression QUESTION expression COLON conditional_expression */
;

assignment_expression:
  conditional_expression    { $1 }
| unary_expression assignment_operator assignment_expression { exp_here (EAssign($2, $1, $3)) }
;

assignment_operator:
  EQUALS          { AssignOp }
| STAREQUALS      { MultAssignOp }
| SLASHEQUALS     { DivAssignOp }
| PERCENTEQUALS   { ModAssignOp }
| PLUSEQUALS      { AddAssignOp }
| MINUSEQUALS     { SubAssignOp }
| LTLTEQUALS      { LShiftAssignOp }
| GTGTEQUALS      { RShiftAssignOp }
| AMPERSANDEQUALS { BitwiseAndAssignOp }
| CARETEQUALS     { XorAssignOp }
| BAREQUALS       { BitwiseOrAssignOp }
;

type_app:
  UID EQUALS typ1   { ($1, $3) }
;

type_app_list:
  type_app                        { [$1] }
| type_app COMMA type_app_list    { $1::$3 }
;

unpack_list: { [] } | unpack_list1 { $1 }
unpack_list1:
  tvar EQUALS UID { [($3, $1)] }
| tvar EQUALS UID COMMA unpack_list1 { ($3, $1)::$5 }
;

postfix_expression:
  primary_expression                    { $1 }
| postfix_expression LBRACKET expression RBRACKET     { raise (InternalError "not implemented: array") }
| postfix_expression LPAREN argument_expression_list RPAREN { exp_here (ECall ($1, $3)) }
| postfix_expression DOT field_id     { exp_here (EMember ($1, $3)) }
/*| postfix_expression RARROW field_id   { exp_here (EMember (exp_here (EDeref $1), $3)) }*/
/*| postfix_expression PLUSPLUS */
/*| postfix_expression MINUSMINUS */
| postfix_expression LBRACKET type_app_list RBRACKET   { exp_here (ETApp ($1, $3)) }
/*| TYPEFUN typ LPAREN expression RPAREN   { exp_here (ETFun ($2, $4)) }*/
;

argument_expression_list: { [] } | argument_expression_list_1 { $1 };
argument_expression_list_1:
  assignment_expression   { [$1] }
| assignment_expression	COMMA argument_expression_list_1 { $1::$3 }
;

unary_expression:
  postfix_expression                  { $1 }
/*| PLUSPLUS unary_expression */
/*| MINUSMINUS unary_expression */
/*| STAR unary_expression               { exp_here (EDeref $2) }*/
/*| AMPERSAND unary_expression */
/*
| unary_operator cast_expression      { exp_here (EUnary ($1, $2)) }
*/
/*	| SIZEOF unary_expression */
/*	| SIZEOF LPAREN type_name RPAREN */
;

/*
unary_operator:
  PLUS    { PositiveOp }
| MINUS   { NegativeOp }
| TILDE   { BitwiseNotOp }
| BANG    { NotOp }
;
*/

cast_expression:
  unary_expression { $1 }
/* | LPAREN type_name RPAREN cast_expression */
;

multiplicative_expression:
  cast_expression   { $1 }
| multiplicative_expression STAR cast_expression      { ebinary MultOp $1 $3 }
/*
| multiplicative_expression SLASH cast_expression     { ebinary DivOp $1 $3 }
| multiplicative_expression PERCENT cast_expression   { ebinary ModOp $1 $3 }
*/
;

additive_expression:
  multiplicative_expression   { $1 }
| additive_expression PLUS multiplicative_expression    { ebinary AddOp $1 $3 }
| additive_expression MINUS multiplicative_expression   { ebinary SubOp $1 $3 }
| MINUS multiplicative_expression                       { ebinary SubOp (exp_here (EInt Big_int.zero_big_int)) $2 }

shift_expression:
  additive_expression     { $1 }
| shift_expression LTLT additive_expression       { ebinary LShiftOp $1 $3 }
| shift_expression GTGT additive_expression       { ebinary RShiftOp $1 $3 }
;

relational_expression:
  shift_expression      { $1 }
| relational_expression LT shift_expression             { ebinary LtOp $1 $3 }
| relational_expression GT shift_expression             { ebinary GtOp $1 $3 }
| relational_expression LE shift_expression             { ebinary LeOp $1 $3 }
| relational_expression GE shift_expression             { ebinary GeOp $1 $3 }
;

equality_expression:
  relational_expression   { $1 }
| equality_expression EQUALSEQUALS relational_expression  { ebinary EqOp $1 $3 }
| equality_expression NE relational_expression            { ebinary NeOp $1 $3 }
;

AND_expression:
  equality_expression     { $1 }
| AND_expression AMPERSAND equality_expression { ebinary BitwiseAndOp $1 $3 }
;

exclusive_OR_expression:
  AND_expression    { $1 }
| exclusive_OR_expression CARET AND_expression { ebinary BitwiseXorOp $1 $3 }
;

inclusive_OR_expression:
  exclusive_OR_expression   { $1 }
| inclusive_OR_expression BAR exclusive_OR_expression { ebinary BitwiseOrOp $1 $3 }
;

logical_NOT_expression:
  inclusive_OR_expression { $1 }
| BANG logical_NOT_expression { eunary NotOp $2 }
;

logical_AND_expression:
  logical_NOT_expression   { $1 }
| logical_AND_expression AMPERSANDAMPERSAND logical_NOT_expression { ebinary AndOp $1 $3 }
;

logical_OR_expression:
  logical_AND_expression    { $1 }
| logical_OR_expression BARBAR logical_AND_expression  { ebinary OrOp $1 $3 }
;

var: ID         { ($1, 0) };
tvar: UID       { ($1, 0) };
/* identifier: ID  { $1 }; */
field_id:
  ID { $1 }
| CONSTINT { Big_int.string_of_big_int $1 }
;

