module PCTypes = struct

  open Ast

  type precompilerInfo =
      {
	expHandlers: expHandler list;
	stmtHandlers: stmtHandler list;
	fcnHandlers: fcnHandler list;
	typHandlers: typHandler list;
	tdHandlers: tdHandler list;
	cdHandlers: cdHandler list;
	finalizers: finalizer list;
	pcFilename: string;
      }

  and ehResult = ExpChanged of exp | ExpUnchanged
  and expHandler = exp -> precompilerInfo -> ehResult

  and shResult = StmtChanged of stmt | StmtUnchanged
  and stmtHandler = stmt -> precompilerInfo -> shResult

  (* Function handlers catch top level function declarations,
   * and can introduce new functions in addition to changing
   * the function they caught
   *)
  and fcnResult = FcnChanged of fun_decl | FcnUnchanged
  and fcnHandler = fun_decl -> precompilerInfo -> (fcnResult * fun_decl list)

  (* type handlers catch type expressions within either type declarations 
   * (such as the right side of a typedef) or within expressions (such as TApps
   * or packs)
   *)
  and thResult = TypChanged of typ | TypUnchanged
  and typHandler = typ -> precompilerInfo -> thResult

  (* type declaration handlers catch top-level type declarations, and are
   * the other handler that can introduce new functions and types in addition
   * to changing the type declaration they caught.
   *)
  and tdResult = TDChanged of type_decl | TDUnchanged
  and tdHandler = type_decl -> precompilerInfo -> (tdResult * type_decl list * fun_decl list)

  (* Compiler directive handlers catch top-level compiler directives, and can
   * return lists of functions and/or types based on that directive if they
   * choose
   *)

  and cdHandler = compiler_directive -> precompilerInfo -> (compiler_directive list * type_decl list * fun_decl list)

  (* finalizers run after the rest of preprocessing is complete, and can do anything
   * they want based on the types and functions passed to them, but cannot change
   * any aspect of the program.  They were added to support exporting interface
   * files for separate compilation.
   *)

  and finalizer = program -> precompilerInfo -> unit

  exception Precompilation of string;;
end;;

module type PCBasis = sig
  val info:PCTypes.precompilerInfo
end;;

module type Precompiler = sig
  val pc: Ast.program -> Ast.program
end;;

module Make (Basis:PCBasis):Precompiler = struct

  open Ast
  open Astiter
  open PCTypes

  module IterBasis = struct
    
    type cd_handler_result = compiler_directive option
    type td_handler_result = type_decl
    type fd_handler_result = fun_decl
    type exp_handler_result = exp
    type stmt_handler_result = stmt
    type typ_handler_result = typ

    type iteration_result = program

    type iters = 
	{
	  iterCDs: compiler_directive list -> (type_decl list * fun_decl list * cd_handler_result list);
	  iterTDs: type_decl list -> (fun_decl list * td_handler_result list);
	  iterFDs: fun_decl list -> (fd_handler_result list);
	  iterExp: exp -> exp_handler_result;
	  iterTyp: typ -> typ_handler_result;
	  iterStmt: stmt -> stmt_handler_result
	}

    let rec handleCD (i:iters) (cd:compiler_directive):(compiler_directive list * type_decl list * fun_decl list * compiler_directive option) =
      let rec recurse (newCDs:compiler_directive list) (newTypes:type_decl list) (newFuncs: fun_decl list) (handlers:cdHandler list):
	                          (compiler_directive list * type_decl list * fun_decl list) =
	match handlers with
	    [] -> (newCDs, newTypes, newFuncs)
	  | (h::hs) ->
	      let (cds, tds, fds) = h cd Basis.info in
		recurse (newCDs @ cds) (newTypes @ tds) (newFuncs @ fds) hs
      in
	match cd with
	    CDEmit emit -> ([], [], [], Some (CDEmit emit))
	  | _ -> (let (cds, tds, fds) = recurse [] [] [] Basis.info.cdHandlers in
		    (cds, tds, fds, None))

    and handleTD (i:iters) ((name, spec):type_decl):(type_decl list * fun_decl list * type_decl) =
      let rec recurse (td:type_decl) (newTypes:type_decl list) (newFuncs:fun_decl list) (handlers:tdHandler list):(type_decl list * fun_decl list * type_decl) =
	match handlers with
	    [] -> (newTypes, newFuncs, td)
	  | (h::hs) -> 
	      (match h td Basis.info with
		   (TDUnchanged, tds, fds) -> recurse td (newTypes @ tds) (newFuncs @ fds) hs
		 | (TDChanged td, _, _) -> recurse td [] [] Basis.info.tdHandlers) in
      let td' =
	match spec with
	    AbbrevSpec t -> (name, AbbrevSpec (i.iterTyp t))
	  | StructSpec (nbits, lin, tp_opt, fields) ->
	      let fields' = List.map (fun (name, t) -> (name, (i.iterTyp t))) fields in
		(name, StructSpec (nbits, lin, tp_opt, fields'))
	  | EnumSpec _ -> (name, spec)
	  | NativeSpec _ -> (name, spec)
      in
	recurse td' [] [] Basis.info.tdHandlers

    and handleFD (i:iters) (fd:fun_decl):(fun_decl list * fun_decl) =
      let rec recurse (fd:fun_decl) (newFuncs:fun_decl list) (handlers:fcnHandler list):(fun_decl list * fun_decl) =
	match handlers with
	    [] -> (newFuncs, fd)
	  | (h::hs) ->
	      (match h fd Basis.info with
		   (FcnUnchanged, fds) -> recurse fd (newFuncs @ fds) hs
		 | (FcnChanged fd', _) -> recurse fd' [] Basis.info.fcnHandlers) in
      let fd' =
	match fd.fun_decl_stmt with
	    FunBody s -> { fd with fun_decl_stmt = FunBody (i.iterStmt s) }
	  | _ -> fd
      in
	recurse fd' [] Basis.info.fcnHandlers

    and handleExp (i:iters) (e:exp):exp =
      let rec recurse (e:exp) (handlers:expHandler list):exp =
	match handlers with
	    [] -> e
	  | (h::hs) ->
	      (match h e Basis.info with
		   ExpUnchanged -> recurse e hs
		 | ExpChanged e' -> recurse e' Basis.info.expHandlers) in

      let er' = 
	match e.exp_raw with
	    ECall (fcn, args) ->
	      ECall (handleExp i fcn, List.map (handleExp i) args)
	  | EAssign (op, arg1, arg2) ->
	      EAssign (op, handleExp i arg1, handleExp i arg2)
	  | ERecord (lin, fields) ->
	      ERecord (lin, List.map (fun (name, typ_opt, e') ->
					(name,
					 (match typ_opt with
					      Some t -> Some (i.iterTyp t)
					    | None -> None),
					 handleExp i e')) fields)
	  | EMember (e', field) -> EMember (handleExp i e', field)
	  | ETApp (e', args) ->
	      ETApp (handleExp i e', List.map (fun (name, t) -> (name, i.iterTyp t)) args)
	  | EPack (e', t, params) ->
	      EPack (handleExp i e', i.iterTyp t, List.map (fun (name, t) -> (name, i.iterTyp t)) params)
	  | _ -> e.exp_raw
      in
	recurse { e with exp_raw = er' } Basis.info.expHandlers

    and handleStmt (i:iters) (s:stmt):stmt =
      let rec recurse (s:stmt) (handlers:stmtHandler list):stmt =
	match handlers with
	    [] -> s
	  | (h::hs) ->
	      (match h s Basis.info with
		   StmtUnchanged -> recurse s hs
		 | StmtChanged s' -> recurse s' Basis.info.stmtHandlers) in

      let sr' = 
	match s.stmt_raw with
	    SBlock stmts -> SBlock (List.map (handleStmt i) stmts)
	  | SDecl (u, ((v, typ_opt), e), s') ->
	      SDecl (u, 
		     ((v, (match typ_opt with Some t -> Some (i.iterTyp t) | None -> None)), i.iterExp e), 
		     handleStmt i s')
	  | SMDecl (u, (vs, e), s') ->
	      SMDecl (u,
		      (List.map (fun (name, typ_opt) -> (name, (match typ_opt with Some t -> Some (i.iterTyp t) | None -> None))) vs,
		       i.iterExp e),
		      handleStmt i s')
	  | SReturn e -> SReturn (i.iterExp e)
	  | SWhile (cond, loop, rest) -> SWhile (i.iterExp cond, handleStmt i loop, handleStmt i rest)
	  | SFor (tparams, ba, init, cond, incr, loop, rest) ->
	      SFor (tparams,
		    ba,
		    List.map (fun ((v, t), e) -> ((v, i.iterTyp t), i.iterExp e)) init,
		    i.iterExp cond,
		    (match incr with
			 Some es -> Some (List.map i.iterExp es)
		       | None -> None),
		    handleStmt i loop,
		    handleStmt i rest)
	  | SContinue es -> SContinue (List.map i.iterExp es)
	  | SIfElse (cond, cons, alt) -> SIfElse (i.iterExp cond, handleStmt i cons, handleStmt i alt)
	  | SBoolCase (ba, cons, alt) -> SBoolCase (ba, handleStmt i cons, handleStmt i alt)
	  | SExp e -> SExp (i.iterExp e)
	  | _ -> s.stmt_raw
      in
	recurse { s with stmt_raw = sr' } Basis.info.stmtHandlers

    and handleTyp (i:iters) (t:typ):typ =
      let rec recurse (t:typ) (handlers:typHandler list):typ =
	match handlers with
	    [] -> t
	  | (h::hs) ->
	      (match h t Basis.info with
		   TypUnchanged -> recurse t hs
		 | TypChanged t' -> recurse t' Basis.info.typHandlers) in

      let t' =
	match t with
	    TArrow (params, result, limit) ->
	      TArrow (List.map (handleTyp i) params, handleTyp i result, limit)
	  | TApp (fcn, args) ->
	      TApp (handleTyp i fcn, List.map (handleTyp i) args)
	  | TRecord (lin, fields) ->
	      TRecord (lin, List.map (fun (name, t) -> (name, handleTyp i t) ) fields)
	  | TExists (tparams, ba, t) -> TExists (tparams, ba, handleTyp i t)
	  | TAll (tparams, ba, t) -> TAll (tparams, ba, handleTyp i t)
	  | TFun (tparams, t) -> TFun (tparams, handleTyp i t)
	  | _ -> t
      in 
	recurse t' Basis.info.typHandlers

    and combine (cdos:compiler_directive option list) (types:type_decl list) (funcs:fun_decl list):program =
      let cds = List.fold_right (fun cd_opt cds -> match cd_opt with Some cd -> cd :: cds | None -> cds) cdos [] in
      let p = { program_directives = cds; program_fun_decls = funcs; program_type_decls = types } in
	List.iter (fun (f:finalizer) -> f p Basis.info) Basis.info.finalizers;
	p

  end

  module Iter = MakeIterator (IterBasis)

  let pc (p:program):program = 
    let p' = { p with program_directives = p.program_directives @ [CDEndOfFile] } in
      Iter.iterProgram p'

end;;
