#if !defined _GMPXX_H
#define _GMPXX_H

#include <stdio.h>
#include <stdlib.h>
#include <gmp.h>
#include <assert.h>

#include <iostream.h>


#define ASSERT(expr) assert(expr)

int sign(int);


/**
 * GMP++ GNU MPz for C++
 * @author Edward Wei
 */
class BigInt {

private:
	mpz_t mp;
	enum INF {NEGINF = -1, NOTINF = 0,  POSINF = 1} infiniteStatus; //Eric has changed NOTINF from -1 to 0

public:
	static const BigInt posInfinity;
	static const BigInt negInfinity;


	BigInt() {

		mpz_init(mp);
		ASSERT(mp->_mp_d != NULL);
		infiniteStatus = NOTINF;
	}
	
	
	~BigInt() {

		//cout << "BigInt Called Clear " << this << endl;
		mpz_clear(mp);
		mp->_mp_d = (mp_limb_t*)0xDEADBEEF;
	}
		
		
	BigInt(signed int b) {
#ifdef DEBUG
	  //cout<<"\n in BigInt::BigInt(signed int)"<<endl;
#endif //DEBUG

		mpz_init_set_si(mp, b);

		ASSERT(mp->_mp_d != NULL);
		infiniteStatus = NOTINF;
#ifdef DEBUG
		//cout<<"\n leaving BigInt::BigInt(signed int)"<<endl;
#endif //DEBUG

	}


	BigInt(unsigned int b) {

		mpz_init_set_ui(mp, b);

		ASSERT(mp->_mp_d != NULL);
		infiniteStatus = NOTINF;
	}


	BigInt(const char* str) {

	        ASSERT(mpz_init_set_str(mp, str, 10) == 0);

		ASSERT(mp->_mp_d != NULL);
		infiniteStatus = NOTINF;
	}

	/**
	 * Copy constructor
	 */
	BigInt(const BigInt& b) {

		mpz_init(mp);

		ASSERT(mp->_mp_d != NULL);
		mpz_set(mp, b.mp);
		infiniteStatus = b.infiniteStatus;
	}


	friend ostream& operator<< (ostream &os,  const BigInt& b);
	friend BigInt operator-(const BigInt& b);
	friend BigInt operator+(int i, const BigInt& b);
	friend BigInt operator*(signed int i, const BigInt& b);
	friend BigInt abs(const BigInt& b);

private:
	static BigInt posInf() {

		BigInt posInfValue;
		posInfValue.infiniteStatus = POSINF;
		return posInfValue;
	}

	static BigInt negInf() {

		BigInt negInfValue;
		negInfValue.infiniteStatus = NEGINF;
		return negInfValue;
	}


// the operators
public:


	//operator bool() const { return (*this) != 0; }

	
	BigInt& operator=(const BigInt &z) {

		mpz_set(mp, z.mp);
		infiniteStatus = z.infiniteStatus;
		return *this;
	}


	BigInt& operator=(unsigned int i) {

		mpz_set_ui(mp, i);
		infiniteStatus = NOTINF;
		return *this;
	}


	BigInt& operator=(signed int i) {

		mpz_set_si(mp, i);
		infiniteStatus = NOTINF;
		return *this;
	}


	char* toCharArray() const {

		return mpz_get_str(NULL, 10, mp);
	}


	bool operator== (const BigInt& b) const {

	  if (infiniteStatus != NOTINF || b.infiniteStatus != NOTINF)
			return infiniteStatus == b.infiniteStatus;
		else
			return mpz_cmp(mp, b.mp) == 0;
	}


	bool operator== (const signed int i) const {

		return (infiniteStatus == NOTINF) && mpz_cmp_si(mp, i) == 0;
	}


	bool operator== (const unsigned int i) const {

		return (infiniteStatus == NOTINF) && mpz_cmp_ui(mp, i) == 0;
	}


	bool operator!= (const BigInt& b) const {

		if (infiniteStatus != NOTINF || b.infiniteStatus != NOTINF)
			return infiniteStatus != b.infiniteStatus;
		else
			return mpz_cmp(mp, b.mp) != 0;
	}


	bool operator!= (const signed int i) const {

		return (infiniteStatus != NOTINF) || mpz_cmp_si(mp, i) != 0;
	}


	bool operator!= (const unsigned int i) const {

		return (infiniteStatus != NOTINF) || mpz_cmp_ui(mp, i) != 0;
	}


	/**
	 * Takes into account infinity
	 */
	bool operator<(const BigInt& b) const {

		// make sure we aren't inf < inf
		//ASSERT(infiniteStatus == NOTINF || (infiniteStatus != b.infiniteStatus));
	  // No, +inf < +inf shoudl return false ERIC
	  if  (infiniteStatus == NOTINF && b.infiniteStatus == NOTINF)
		return (mpz_cmp(mp, b.mp) < 0);
	  else return (infiniteStatus < b.infiniteStatus);
	}


	/**
	 * Takes into account infinity
	 */
	bool operator<=(const BigInt& b) const {

		// make sure we aren't inf < inf
	        //ASSERT(infiniteStatus == NOTINF || (infiniteStatus != b.infiniteStatus));
	  // No, inf < inf should return false ERIC
	  if  (infiniteStatus == NOTINF && b.infiniteStatus == NOTINF)
		return (mpz_cmp(mp, b.mp) <= 0);
	  else return (infiniteStatus <= b.infiniteStatus);
	}


	/**
	 * Takes into account infinity
	 */
	bool operator>(const BigInt& b) const {

		// make sure we aren't inf > inf
		//ASSERT(infiniteStatus == NOTINF || (infiniteStatus != b.infiniteStatus));
	  // No, inf < inf shoudl return false ERIC
	  if  (infiniteStatus == NOTINF && b.infiniteStatus == NOTINF)
		return (mpz_cmp(mp, b.mp) > 0);
	  else return (infiniteStatus > b.infiniteStatus);
	}


	/**
	 * Takes into account infinity
	 */
	bool operator>=(const BigInt& b) const {

		// make sure we aren't inf > inf
	        // ASSERT(infiniteStatus == NOTINF || (infiniteStatus != b.infiniteStatus));
	  // No, inf < inf shoudl return false ERIC
	  if  (infiniteStatus == NOTINF && b.infiniteStatus == NOTINF)
		return (mpz_cmp(mp, b.mp) >= 0);
	  else return (infiniteStatus >= b.infiniteStatus);
	}


	/**
	 * Not allowed to add infinite values
	 */
	BigInt operator+(const BigInt& b) const {

		// we cannot inf + inf
		ASSERT(infiniteStatus == NOTINF);
		ASSERT(b.infiniteStatus == NOTINF);
BigInt result; mpz_add(result.mp, mp, b.mp); return result;
	}


	/**
	 * Not allowed to add infinite values
	 */
	BigInt operator+(unsigned int i) const {

		// we cannot inf + inf
		ASSERT(infiniteStatus == NOTINF);

		BigInt result;
		mpz_add_ui(result.mp, mp, i);

		return result;
	}


	/**
	 * Not allowed to add infinite values
	 */
	BigInt operator+(signed int i) const {

		// we cannot inf + inf
		ASSERT(infiniteStatus == NOTINF);

		BigInt result;

		if (i > 0)
			mpz_add_ui(result.mp, mp, i);
		else
			mpz_sub_ui(result.mp, mp, abs(i));

		return result;
	}



	/**
	 * Not allowed to subtract infinite values
	 */
	BigInt operator-(const BigInt& b) const {

		// we cannot inf - inf
		ASSERT(infiniteStatus == NOTINF);
		ASSERT(b.infiniteStatus == NOTINF);

		BigInt result;
		mpz_sub(result.mp, mp, b.mp);

		return result;
	}


	/**
	 * Not allowed to subtract infinite values
	 */
	BigInt operator-(unsigned int i) const {

		// we cannot inf - inf
		ASSERT(infiniteStatus == NOTINF);

		BigInt result;
		mpz_sub_ui(result.mp, mp, i);

		return result;
	}


	/**
	 * Not allowed to subtract infinite values
	 */
	BigInt operator-(signed int i) const {

		// we cannot inf - inf
		ASSERT(infiniteStatus == NOTINF);

		BigInt result;

		if (i > 0)
			mpz_sub_ui(result.mp, mp, i);
		else
			mpz_add_ui(result.mp, mp, abs(i));

		return result;
	}


	BigInt& operator++() {

		ASSERT(infiniteStatus == NOTINF);
		mpz_add_ui(mp, mp, 1);
		return *this;
	}


	BigInt& operator++(int) {

		ASSERT(infiniteStatus == NOTINF);
		mpz_add_ui(mp, mp, 1);
		return *this;
	}

	
	BigInt& operator+=(const BigInt& b) {

		ASSERT(infiniteStatus == NOTINF);
		mpz_add(mp, mp, b.mp);

		return *this;
	}


	BigInt& operator+=(signed int i) {

		ASSERT(infiniteStatus == NOTINF);
		if (i > 0)
			mpz_add_ui(mp, mp, i);
		else
			mpz_sub_ui(mp, mp, abs(i));

		return *this;
	}


	BigInt& operator+=(unsigned int i) {

		ASSERT(infiniteStatus == NOTINF);
		mpz_add_ui(mp, mp, i);

		return *this;
	}


	BigInt& operator--() {

		ASSERT(infiniteStatus == NOTINF);
		mpz_sub_ui(mp, mp, 1);
		return *this;
	}


	BigInt& operator--(int) {

		ASSERT(infiniteStatus == NOTINF);
		mpz_sub_ui(mp, mp, 1);
		return *this;
	}


	BigInt& operator-=(const BigInt& b) {

		ASSERT(infiniteStatus == NOTINF);
		mpz_sub(mp, mp, b.mp);

		return *this;
	}


	BigInt& operator-=(signed int i) {

		ASSERT(infiniteStatus == NOTINF);

		if (i > 0)
			mpz_sub_ui(mp, mp, i);
		else
			mpz_add_ui(mp, mp, abs(i));

		return *this;
	}


	BigInt& operator-=(unsigned int i) {

		ASSERT(infiniteStatus == NOTINF);
		mpz_sub_ui(mp, mp, i);

		return *this;
	}


	/**
	 * ok... the inf check is b/c omega does sign * infinity
	 */
	BigInt operator*(const BigInt& b) const {

		// we cannot do inf * inf
		ASSERT(infiniteStatus == NOTINF || b.infiniteStatus == NOTINF);
		
		BigInt result;
		
		if (infiniteStatus != NOTINF && b.infiniteStatus == NOTINF)
			result.infiniteStatus = INF(infiniteStatus * (mpz_sgn(b.mp)));
		else if (infiniteStatus == NOTINF && b.infiniteStatus != NOTINF)
			result.infiniteStatus = INF(b.infiniteStatus * (mpz_sgn(mp)));
		else
			mpz_mul(result.mp, mp, b.mp);

		return result;
	}


	/**
	 * ok... the inf check is b/c omega does sign * infinity
	 */
	BigInt operator*(signed int i) const {

		BigInt result;
		
		if (infiniteStatus != NOTINF)
			result.infiniteStatus = INF(infiniteStatus * sign(i));
		else
			mpz_mul_si(result.mp, mp, i);

		return result;
	}


	BigInt& operator*=(BigInt& b) {

		ASSERT(infiniteStatus == NOTINF);
		mpz_mul(mp, mp, b.mp);
		return *this;
	}


	// are we sure about this one?
	BigInt& operator*=(signed int i) {

		ASSERT(infiniteStatus == NOTINF);
		mpz_mul_si(mp, mp, i);

		return *this;
	}


	BigInt& operator*=(unsigned int i) {

		ASSERT(infiniteStatus == NOTINF);
		mpz_mul_ui(mp, mp, i);

		return *this;
	}


	BigInt operator%(const BigInt& b) const {

		ASSERT(infiniteStatus == NOTINF && b.infiniteStatus == NOTINF);

		BigInt result;
		mpz_tdiv_r(result.mp, mp, b.mp);

		return result;
	}


	BigInt operator%(unsigned int i) const {

		ASSERT(infiniteStatus == NOTINF);

		BigInt result;
		mpz_tdiv_r_ui(result.mp, mp, i);

		return result;
	}


	BigInt operator%(signed int i) const {

		// TODO: what is mod neg number?
		ASSERT(infiniteStatus == NOTINF && i >= 0);

		BigInt result;
		mpz_tdiv_r_ui(result.mp, mp, i);

		return result;
	}


	/**
	 * This just gets the quotient.
	 * Do not allow any inf divisions
	 */
	BigInt operator/(const BigInt& b) const {

		ASSERT(infiniteStatus == NOTINF && b.infiniteStatus == NOTINF);

		BigInt result;
		mpz_tdiv_q(result.mp, mp, b.mp);

		return result;
	}


	BigInt& operator/=(BigInt& b) {

		ASSERT(infiniteStatus == NOTINF && b.infiniteStatus == NOTINF);
		mpz_tdiv_q(mp, mp, b.mp);

		return *this;
	}


	BigInt& operator/=(signed int i) {

		ASSERT(infiniteStatus == NOTINF);

		if (i > 0)
			mpz_tdiv_q_ui(mp, mp, i);
		else {
			mpz_tdiv_q_ui(mp, mp, abs(i));
			mpz_neg(mp, mp);
		}

		return *this;
	}


	BigInt& operator/=(unsigned int i) {

		ASSERT(infiniteStatus == NOTINF);
		mpz_tdiv_q_ui(mp, mp, i);

		return *this;
	}


	int get_sint() const {

		ASSERT(mpz_fits_sint_p(mp) != 0);

		return (int)mpz_get_si(mp);
	}


};




/******
 * Global Functions
 *********/

inline ostream& operator<< (ostream &os,  const BigInt& b) {

	if (b.infiniteStatus == BigInt::POSINF)
		return os << "posInf";
	else if (b.infiniteStatus == BigInt::NEGINF)
		return os << "negInf";
	else {
		// printing in base 10 right now.
		return os << mpz_get_str(NULL, 10, b.mp);
	}
}


/**
 * Neg Prefix
 */
inline BigInt operator-(const BigInt& b) {

	BigInt result;

	if (b.infiniteStatus != BigInt::NOTINF)
		result.infiniteStatus = BigInt::INF(-b.infiniteStatus);
	else
		mpz_neg(result.mp, b.mp);

	return result;
}


inline bool operator!(const BigInt& b) {

	return b == 0;
}


inline bool operator&&(const BigInt& i, bool b) {

	return (i != 0 && b);
}


inline bool operator&&(bool b, const BigInt& i) {

	return (i != 0 && b);
}


inline BigInt abs(const BigInt& b) {

	BigInt result;
	if (b.infiniteStatus != BigInt::NOTINF)
		result.infiniteStatus = BigInt::POSINF;
	else
		mpz_abs(result.mp, b.mp);

	return result;
}


inline BigInt operator+(signed int i, const BigInt& b) {

	ASSERT(b.infiniteStatus == BigInt::NOTINF);
	
	BigInt result;
	if (i > 0)
		mpz_add_ui(result.mp, b.mp, i);
	else
		mpz_sub_ui(result.mp, b.mp, abs(i));

	return result;
}


inline BigInt operator*(signed int i, const BigInt& b) {

	return BigInt(b*i);
}


/**
 * For i >= b ?
 */
inline bool operator>=(signed int i, const BigInt& b) {

	return b<=i;
}


inline bool operator<=(signed int i, const BigInt& b) {

	return b>=i;
}


/**
 * Helper to return sign of int
 */
inline int sign(signed int i) {

	if (i > 0)
		return 1;
	else if (i < 0)
		return -1;
	else
		return 0;
}




#endif // defined _GMPXX_H
