/* 
 * Laddie to Clay Compiler
 * Author: Lea Wittie and Derrin Pierret
 * Copyright (c) 2007 Bucknell University
 *
 * Permission is hereby granted, free of charge, to any individual or
 * institution obtaining a copy of this software and associated
 * documentation files (the "Software"), to use, copy, modify, and
 * distribute without restriction, provided that this copyright and
 * permission notice is maintained, intact, in all copies and supporting
 * documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL BUCKNELL UNIVERSITY BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

using namespace std;
#include <iostream>
#include <string>
#include "compiler.h"
#include "support.h"
#include "Generate.h"
#include <vector>
#include <fstream>
#include <sstream>

//#define DEBUG
#ifdef DEBUG
#define err(s) cout << s << endl;
#else 
#define err(s)
#endif

// clears globally accessed info on the current port
void Generate::resetPortInfo() {
  int i;
  fieldCount = 0;
  fieldNames.resize(0);
  fieldSizes.resize(0);
  fieldLetters.resize(0);
  wereSwitchCases = false;
  for(i=0; i<stateCount; i++) {
    usedStates.at(i) = false;
    changedStates.at(i) = false;
  }
  mode = 0;
  io = "";
}

Generate::Generate() {
  // Set any local variables here
  stateCount = 0;
  stateNames.resize(0);
  stateSizes.resize(0);
  stateLetters.resize(0);
  IOCommands = new SymbolTable();
  resetPortInfo();
    
  //IOCommands = new SymbolTable();
}

void Generate::walkTree(TreeNode* nodePtr) {
  string tempstr;

  // ----------- Open files
  tempstr = fname + "Native.clh";
  outNative.open(tempstr.c_str(), fstream::out);

  tempstr = fname + "NativeIO.clh";
  outNativeIO.open(tempstr.c_str(), fstream::out);

  tempstr = fname + "IO.clh";
  outFunction.open(tempstr.c_str(), fstream::out);

  tempstr = fname + "Macro.clh";
  outMacro.open(tempstr.c_str(), fstream::out);

  //tempstr = fname + "Defines.h";
  //outDefines.open(tempstr.c_str(), fstream::out);

  // ----------- Generate tests
  if (consistency) {
    cerr << "Generating consistency tests and script.\n"
	 << "To run, type\n"
	 << "\tchmod a+x " << fname << "_testscript\n"
	 << "\t" << fname << "_testscript\n"
	 << "There will be no output with consistent conditions.\n";
    cons_constraint = "12345>12348";
    makeConsistencyChecker();
    // tested: we can trap the output of Clay into a file too.
  }

  if (translation) {
    makeTest();
    cerr << "Generating translation test. \n"
	 << "This test looks for Clay syntax, and typing errors.\n"
	 << "To run, type\n"
	 << "\tclay " << fname << "Test.clay\n"
	 << "\tg++ " << fname << "Main.cc " << fname << "Test.cc\n"
	 << "There will be no output on a correct translation.\n"
	 << "To test specification consistency, use the -consistency flag.\n";
  }
  
  // ----------- Generate code
  driver(nodePtr);
    
  // ----------- Close files
  outNative.close();
  outFunction.close();
  outMacro.close();
  //outDefines.close();
}

// Generate files to test the consistency of the Laddie specification
void Generate::makeTest() {
  string tempstr;

  tempstr = fname + "Test.clay";
  outTest.open(tempstr.c_str(), fstream::out);
  outTest << "// This is a translation test for the Laddie spec.\n\n"
	  << "#include \"std.clh\"\n"
	  << "#include \"" << fname << "IO.clh\"\n"     // clay port access
	  << "#include \"" << fname << "Macro.clh\"\n"   // clay macros
    	  << "#include \"" << fname << "NativeIO.clh\"\n" // extern io calls
    	  << "#include \"" << fname << "Native.clh\"\n"   // native port access

	  << "void test() { }\n";
  outTest.close();

  tempstr = fname + "Main.cc";
  outMain.open(tempstr.c_str(), fstream::out);
  outMain //<< "#include \"" << fname << "Defines.h\"\n"  // actual io calls
	  << "unsigned long read1byte(unsigned long addr) { return 0; }\n"
	  << "unsigned long read2byte(unsigned long addr) { return 0; }\n"
	  << "unsigned long read4byte(unsigned long addr) { return 0; }\n"
	  << "void write1byte(unsigned long addr, unsigned long value) "
	  << "\n  { return; }\n"
	  << "void write2byte(unsigned long addr, unsigned long value) "
	  << "\n  { return; }\n"
	  << "void write4byte(unsigned long addr, unsigned long value) "
	  << "\n  { return; }\n"
	  << "void read1byterep(unsigned long addr, unsigned long count, "
	  << "\n  unsigned long buffer) { return; }\n"
	  << "void read2byterep(unsigned long addr, unsigned long count, "
	  << "\n  unsigned long buffer) { return; }\n"
	  << "void read4byterep(unsigned long addr, unsigned long count, "
	  << "\n  unsigned long buffer) { return; }\n"
	  << "void write1byterep(unsigned long addr, unsigned long count, "
	  << "\n  unsigned long buffer) { return; }\n"
	  << "void write2byterep(unsigned long addr, unsigned long count, "
	  << "\n  unsigned long buffer) { return; }\n"
	  << "void write4byterep(unsigned long addr, unsigned long count, "
	  << "\n  unsigned long buffer) { return; }\n"
	  << "void test();\n"
	  << "int main() {\n  test();\n  return 0;\n}\n";
  outMain.close();
}

/* 1) driver :	states_opt ports */
void Generate::driver(TreeNode* nodePtr) {
  SemanticRec* nextRec;

  // retrieve the current scope
  scope = nodePtr->symTabPtr;

  // ----------- Get state information
  stateCount = 0;
  currName.reset();
  scope->startIterator();

  // get stateCount, stateSizes, stateNames
  while (scope->moreToIterate()) {
    nextRec = scope->nextIteratedSR();
    if (nextRec->kind == STATEINFO) {
      stateCount++;
      stateSizes.resize(stateCount);
      stateNames.resize(stateCount);
      if (nextRec->type == type__1)
	stateSizes.at(stateCount-1) = 1;
      else
	stateSizes.at(stateCount-1) = 32;
      stateNames.at(stateCount-1) = nextRec->key;
    }
  }

  // get stateLetters
  stateLetters.resize(stateCount);
  for(int i = 0 ; i<stateCount ; i++) {
    stateLetters.at(i) = currName.getCurrName();
    currName.genNextName();
  }

  // set vector sizes for later use
  usedStates.resize(stateCount);
  changedStates.resize(stateCount);

  // ----------- Generate state declarations
  err("Gen state types");
  // Declare a type for the base state
  outFunction << "// Linear-size 0 states for this device\n"
	      << "// State names should recognizably match those in the spec.\n\n"
	      << "@type0 " << getStateType("Base") 
	      << " [int Device, int Val] = native" 
	      << endl ;

  // print each states type declaration
  scope->startIterator();
  while (scope->moreToIterate()) {
    nextRec = scope->nextIteratedSR();
    if (nextRec->kind == STATEINFO) {
      outFunction << "@type0 " <<  getStateType(nextRec->key)
		  << " [int Device, int Val] = native\n";
    }
  }
  outFunction << endl
	      << "// Port IO function headers "
	      << "(actual C functions in Native.clh)\n"
	      << "// type variables are declared as\n"
	      << "//   V (value read or written)\n"
	      << "//   D (device memory address)\n"
	      << "//   A (base address)\n"
	      << "//   B0in (input state variable)\n"
	      << "//   B0out (output state variable)\n"
	      << "//   C0 (local field)\n"
	      << "// The local and state type variables are named [A-Z][0-99]\n\n";

  // ----------- Generate state and ioaddr #define stubs
  //outDefines << "#ifndef LADDIE_" << fname << "DEFINES_H" << endl
  //	    << "#define LADDIE_" << fname << "DEFINES_H\n" << endl
  //	    << "#define IOADDR addr \n" << endl
  
    //outDefines << "native {\n"
  //	    << "/*\n User may need to define the io"
  //	    << "functions used by generated port functions." 
  //	    << "\n For example, '#define inb(addr) myreadfunction(addr)'.\n*/"
  //	    << endl;

  //outNative << "#ifndef LADDIE_" << fname << "NATIVE_H" << endl
  //	    << "#define LADDIE_" << fname << "NATIVE_H" << endl
  //	    << "#include \"" << fname << "Defines.h\"" << endl
  //	    << "#include \"native.h\"" << endl;
    
  outNative << "native {\n";

  //user def field declaration is done, userDefCount doesnt change
  //userDefIn.resize(stateCount); // keep??
  err("Gen IO");
  ports(nodePtr->ptr2);
   
  outNativeIO << "native {\n";
  IOCommands->startIterator();
  while(IOCommands->moreToIterate()) {
    outNativeIO << "extern " << IOCommands->nextIteratedSR()->name << ";\n";
    //outDefines << IOCommands->nextIteratedSR()->name << 
  }
  outNativeIO << "}\n";

  outNative << "}\n";
  //outDefines << "}\n";

  //outNative << "#endif" << endl;
  //outDefines << "#endif" << endl;
}

/****************************** ports ***********************************/

/* 2) ports :	port ports | port */
void Generate::ports(TreeNode* nodePtr) {
  if (nodePtr->kind == ports__1) {
    port(nodePtr->ptr1);
    ports(nodePtr->ptr2);
  }
  else //if (nodePtr->kind == ports__2) {
    port(nodePtr->ptr1);
}


/* general things */
/*
  num			:	INT | HEX ;
  num_or_set		:	num | '[' constraint_set ']' ;
  num_or_range		:	num | num ':' num ;


  /*
  type			:	BOOLEAN | INTEGER ;
  constraint_set		:	num_or_range
  |	num_or_range ',' constraint_set
*/

/* global state declarations with constraints  */

/* 18) states_opt :	  | states */
/* 19) states : state | state states */
/* state		: type ID ';' 
 *   | type ID relop num_or_set ';' 
 */

/*********************** ports *********************************/

/* 3) port :	PORT  num '{' port_info '}' */
void Generate::port(TreeNode* nodePtr) {
  //cerr << "port " << nodePtr->strval << endl;
  err("  in port "); err(nodePtr->strval);
  SemanticRec* nextRec;
  //cerr << "\nPort " << nodePtr->strval << endl;

  // retrieve the current scope
  scope = nodePtr->symTabPtr;
  currPort = scope->lookup(nodePtr->strval);
 
#ifdef DEBUG
  if (scope == NULL || currPort == NULL) {
    cerr << "Environment Error: Scope information missing in port()\n";
    exit(1);
  }
#endif

  err("  getting field info");
  // ----------- Get field information

  // get fieldCount, fieldSizes, fieldNames
  fieldCount = 0;
  scope->startIterator();
  while (scope->moreToIterate()) {
    nextRec = scope->nextIteratedSR();
    if (nextRec->kind == FIELDINFO
      	&& nextRec->key.substr(0,12)!="__reserved__") {
      
      fieldCount++;
      fieldSizes.resize(fieldCount);
      fieldNames.resize(fieldCount);
      fieldSizes.at(fieldCount-1) = 
	nextRec->range.getRange(0).getHigh() 
	- nextRec->range.getRange(0).getLow();
      fieldNames.at(fieldCount-1) = nextRec->key;
    }
  }    
  //cerr << "  fieldcount " << fieldCount << endl;
  // get fieldLetters
  currName.reset();
  fieldLetters.resize(fieldCount);
  // Run thru the state letters
  for(int i = 0 ; i<stateCount ; i++)
    currName.genNextName();
  // Generate the field letters
  for(int i = 0 ; i<fieldCount ; i++) {
    fieldLetters.at(i) = currName.getCurrName();
    currName.genNextName();
  }
  
  err("  generating conditions");
  port_info(nodePtr->ptr2);  
    
  // print IO Clay header
  //sendPortToFile(); keep??

  // clear member data on the port so we are ready for the next one
  resetPortInfo();
}


/* 5) port_info : name_info size_info_opt type_info_opt access_info_opt 
 *                field_list_opt before_opt after_opt 
 * Decide what (read/write) to generate
 */
void Generate::port_info(TreeNode* nodePtr) {

  env = true;
  io  = "in";
  err("  env - before opt");
  before_opt(nodePtr->ptr6);
  io = "out";
  err("  env - after opt");
  after_opt(nodePtr->ptr7);
  beforeTree = nodePtr->ptr6;
  afterTree = nodePtr->ptr7;
  env = false;

  if (currPort->access == read_write__1) { // read
    makeRead();
  }
  else if (currPort->access == read_write__2) { // write
    makeWrite();
  }
  else {                                // readwrite
    makeRead();
    makeWrite();
  }
}    
void Generate::makeRead() {
  if (currPort->type == 1)
    createIORepeatedFunction(read_write__1);
  else
    createIOReadFunction();
  createNativeReadFunction();
  createMacro("read");
  makeConsistency("read");
}
void Generate::makeWrite() {
  if (currPort->type == 1)
    createIORepeatedFunction(read_write__2);
  else
    createIOWriteFunction();
  createNativeWriteFunction();
  createMacro("write");
  makeConsistency("write");
}
// Generate a consistency test for the current port
void Generate::makeConsistency(string rw) {
  //cerr << "test input length is " << currPort->length << endl;
  if (!consistency) return; // no testing
  if (currPort->length < 1) return; // no test or no test input

  string tempstr;
  SemanticRec* temp;
  Range currRange;
  
  tempstr = "/tmp/" + fname + rw + currPort->name + ".clay";
  outTest.open(tempstr.c_str(), fstream::out);
  outTest << "// This is a translation test for the Laddie spec.\n\n"
	  << "#include \"std.clh\"\n"
	  << "#include \"" << fname << "IO.clh\"\n"
	  << "#include \"" << fname << "Macro.clh\"\n"
	  << "#define IOADDR addr\n"
	  << "native void doh [int X; "
	  <<   cons_constraint << "] (Int[X] x);\n"
	  << "void test[u32 A, u32 D";

  // type-variable declarations (all used states)
  for(int i = 0 ; i<stateCount ; i++)
    if (usedStates.at(i))
      outTest << ", u" << stateSizes.at(i)
	      << " " << stateLetters.at(i);

  outTest << ";\n  true";
 
  // test values (states mentioned in test input)
  for (int i=0; i<currPort->length; i++) {
    temp = scope->lookup(currPort->ids.at(i));
    if (temp != NULL && temp->kind == STATEINFO)
      outTest << " && " << getStateLetter(currPort->ids.at(i)) 
	      << "==" << currPort->values.at(i);
  }

  // global constraints (all used states)
  for(int i = 0 ; i<stateCount ; i++)
    if (usedStates.at(i)) {
      temp = scope->lookup(stateNames.at(i));
      if (!temp->range.isEmpty()) {
	outTest << " && (";

	// Print each part of the constraint
	for(int m = 0 ; m < temp->range.getSize() ; m++) {
	  currRange = temp->range.getRange(m);

	  // || between possibilities "G==2 || G==4"
	  if(m > 0) outTest << " || ";

	  // A value "G==3"
	  if(currRange.getLow() == currRange.getHigh())
	    outTest << stateLetters.at(i)
			<< "==" << currRange.getLow();
    
	  // A range "(0<=G && G<=4)"
	  else 
	    outTest << "(" << stateLetters.at(i)
		    << ">=" << currRange.getLow()
		    << " && " << stateLetters.at(i)
		    << "<=" << currRange.getHigh() <<")"; 
	}

	outTest << ")";
      }
    }

  outTest << "]\n(StateType_Base[D,A] stateVar_Base, Int[A] addr";
  
  for(int i = 0 ; i<stateCount ; i++)
    if (usedStates.at(i))
      outTest << ",\n  " << getStateType(stateNames.at(i)) 
	      << "[D," << stateLetters.at(i) 
	      << "] " << getStateVar(stateNames.at(i));

  outTest << ") {\n";

  if (rw=="read") {
    outTest << "  ";
    createFunctionName("R", outTest);
    outTest << "(";

    if(fieldCount > 0)
      for(int i = 0 ; i < fieldCount ; i++) {
	if (i != 0) outTest << ", ";
	outTest << "f_" << fieldNames.at(i);
      }
  
    else outTest << "value";

    outTest << ");\n";
  }
  else { // write - only on 0 for now
    outTest << "  ";
    createFunctionName("W", outTest);
    outTest << "(";

    // for each field, get its name. find it in the test list. use that value
    if(fieldCount > 0)
      for(int i = 0 ; i < fieldCount ; i++) {
	if (i != 0) outTest << ", ";
	int k;
	for (k=0; 
	     k<currPort->length && currPort->ids.at(k) != fieldNames.at(i); 
	     k++) ;
	// missing means use 0
	if (k==currPort->length)
	  outTest << "0";
	else
	  outTest << currPort->values.at(k);
      }
  
    else { // same for value (no fields)
      int k;
      for (k=0; 
	   k<currPort->length && currPort->ids.at(k) != "value"; 
	   k++) ;
      // missing means use 0
      if (k==currPort->length)
	outTest << "0";
      else
	outTest << currPort->values.at(k);
    }
    outTest << ");\n";
  }
  
  outTest << "  doh(3); // passes if false is in the system\n"
	  << "}\n";
  outTest.close();  

  //  cons_list.pushback(tempstr);
}

void Generate::makeConsistencyChecker() {
  string tempstr;

  tempstr = fname + "_testscript";
  outTest.open(tempstr.c_str(), fstream::out);

  outTest << "#!/usr/bin/perl\n"
	  << "use File::Copy;\n"

	  << "$filetobecopied1 = \"std.clh\";\n"
	  << "$filetobecopied2 = \"" << fname << "IO.clh\";\n"
	  << "$filetobecopied3 = \"" << fname << "Macro.clh\";\n"

	  << "$dest1 = \"/tmp/std.clh\";\n"
	  << "$dest2 = \"/tmp/" << fname << "IO.clh\";\n"
	  << "$dest3 = \"/tmp/" << fname << "Macro.clh\";\n"

	  << "copy($filetobecopied1, $dest1) or die \"No std.clh copy.\";\n"
	  << "copy($filetobecopied2, $dest2) or die \"No IO copy.\";\n"
	  << "copy($filetobecopied3, $dest3) or die \"No Macro copy.\";\n"

	  << "chdir (\"/tmp\") || die \"can't find /tmp, better panic\";\n"
	  << "system(\"rm -f /tmp/*.out\");\n"

	  << "$tmp_dir = \"/tmp\";\n"
	  << "opendir(DIR, $tmp_dir) || die \"Can't read $tmp_dir: $!\";\n"
	  << "@contents = glob (\"" << fname << "*.clay\"), readdir(DIR);\n"
	  << "closedir DIR;\n"

	  << "foreach $listitem ( @contents )\n"
	  << "{\n"
	  << "  system (\"clay $listitem > $listitem.out\");\n"
	  << "  $filename = $listitem . '.out';\n"

	  << "  open (CLAYFILE, $filename) || die \"No out file open.\";\n"
	  << "  $filesize = -s CLAYFILE;\n"

	  << "  if ($filesize == 70) {\n"
 	  << "     system(\"echo $listitem : Impossible postcondition\");\n"
	  << "  }\n"

	  << "  else {\n"
	  << "    @file_arr = <CLAYFILE>;\n"
	  << "    $file_str = \"\";\n"
	  << "    foreach $fileline (@file_arr) {\n"
	  << "	    $file_str = $file_str . $fileline;\n"
	  << "    }\n"

	  << "    if ($file_str =~ m/" << cons_constraint << "/) {\n"
	  << "    } else {\n"
	  << "	    system(\"echo $listitem : Impossible precondition\");\n"
	  << "    }\n"
	  << "  }\n"

	  << "  close (CLAYFILE);\n"

	  << "}\n"
	  << "exit;\n";

  outTest.close();
}

/******************** Print CONSTRAINTS ON I/O **************************
 * Generate IO constraints for reading or writing this function
 * see if it's for R/W/RW
 * get the switch ID if any (already done since general for port?)
before_opt       : | before before_opt ;
before           : BEFORE constraints | BEFORE read_write constraints
         | BEFORE '{' constraints '}' | BEFORE read_write '{' constraints '}' ;
after_opt        : | after after_opt ;
after            : AFTER constraints | AFTER read_write constraints 
         | AFTER '{' constraints '}' | AFTER read_write '{' constraints '}' ;
constraints      : transitions 
                 | SWITCH ID '{' switch_cases default_case_opt '}' ;
*/
#define OPMAX 5


/* 28) before_opt : | before before_opt */
void Generate::before_opt(TreeNode * nodePtr) {
  err("before_opt\n");
  if (nodePtr->kind == before_opt__1) {
    err("  done\n"); 
  }
  else if (nodePtr->kind == before_opt__2) {
    before(nodePtr->ptr1);
    before_opt(nodePtr->ptr2);
  }
  else {
    err("error in before_opt"); err(nodePtr->kind);
  }
}

/* 29) before : BEFORE constraints | BEFORE read_write constraints
   | BEFORE '{' constraints '}' | BEFORE read_write '{' constraints '}' */
void Generate::before(TreeNode* nodePtr) {
  if (nodePtr->kind == before__1 || nodePtr->kind == before__3) {
    constraints(nodePtr->ptr1);
  }
  else {
    if (env || nodePtr->ptr1->kind == mode 
	|| nodePtr->ptr1->kind == read_write__3)
      constraints(nodePtr->ptr2);
  }
}

/* 30) after_opt */
void Generate::after_opt(TreeNode* nodePtr) {
  if (nodePtr->kind == after_opt__1) {
  }
  else {//if (nodePtr->kind == after_opt__2) {
    after(nodePtr->ptr1);
    after_opt(nodePtr->ptr2);
  }
}

/* 31) after */
void Generate::after(TreeNode* nodePtr) {
  if (nodePtr->kind == after__1 || nodePtr->kind == after__3) {
    constraints(nodePtr->ptr1);
  }
  else {
    if (env || nodePtr->ptr1->kind == mode 
	|| nodePtr->ptr1->kind == read_write__3)
      constraints(nodePtr->ptr2);
  }
}

/* 32) constraints : transitions 
   | SWITCH ID '{' switch_cases default_case_opt '}' */
void Generate::constraints(TreeNode* nodePtr) {
  err("constraints");
  if (!env && optCnt%OPMAX == 0)outFunction << "\n    ";
  if (!env && optCnt > 0) { 
    outFunction << " && ";
  }
  if (!env) outFunction << "(";
  if (nodePtr->kind == constraints__1) {
    err("  going to trans");
    transitions(nodePtr->ptr1);
  }
  else if (nodePtr->kind == constraints__2) {
    err("  going to switch");
    err(nodePtr->ptr1->strval);
    if (!env) switchName = nodePtr->ptr1->strval;
    switch_cases(nodePtr->ptr2);
    default_case_opt(nodePtr->ptr3);
  }
  else { 
    err("no constraints kind"); err(nodePtr->kind);
    err(constraints__1);
  }
  //else treeError("constraints");
  if (!env) outFunction << ")";
}

/*
 * switch_cases : switch_case | switch_case switch_cases ;
 * switch_case      : num ':' transitions  | ID ':' transitions
 * default_case_opt : | DEFAULT ':' transitions ;
 */

/* 33) switch_cases */
void Generate::switch_cases(TreeNode* nodePtr) {
  if (!env) wereSwitchCases = true;
  if (nodePtr->kind == switch_cases__1) {
    switch_case(nodePtr->ptr1);
  }
  else {//if (nodePtr->kind == switch_cases__2) {
    switch_case(nodePtr->ptr1);
    if (!env && optCnt%OPMAX == 0) outFunction << "\n    ";
    if (!env) { 
      outFunction << " && ";
    }
    switch_cases(nodePtr->ptr2);
  }
  //else treeError("switch_cases");
}

/* 34) switch_case */
void Generate::switch_case(TreeNode* nodePtr) {
  err("switch case");
  if (nodePtr->kind == switch_case__1) {
    err("  case int");
    if (!env) {
      outFunction << "((" << getFieldLetter(switchName) << "==" 
		  << nodePtr->ptr1->ptr1->numval << " && "; optCnt++;
    }

    // the case
    transitions(nodePtr->ptr2);
    if (!env)  {
      outFunction << ") || " << getFieldLetter(switchName) << "!=" 
		  << nodePtr->ptr1->ptr1->numval << ")"; optCnt++;
    }
  }
  else if (nodePtr->kind == switch_case__2) {
    err("  case id");
    SemanticRec* tmp = scope->lookup(nodePtr->ptr1->strval);
    int num = tmp->num;
    
    if (!env) {
      outFunction << "((" << getFieldLetter(switchName) << "==" 
		  << num << " && "; optCnt++;
    }

    // the case
    transitions(nodePtr->ptr2);
    if (!env)  {
      outFunction << ") || " << getFieldLetter(switchName) << "!=" 
		  << num << ")"; optCnt++;
    }
  }
  else {
    err("error in switch case"); err(nodePtr->kind);
  }
  //else treeError("switch_case");
}

// warning: generated default may be incorrect
/* 35) default_case_opt */
void Generate::default_case_opt(TreeNode* nodePtr) {
  if (nodePtr->kind == default_case_opt__1) {
  }
  else {//if (nodePtr->kind == default_case_opt__2) {
    if (!env && optCnt%OPMAX == 0) outFunction << "\n    ";
    if (!env && wereSwitchCases) {
      outFunction << " || ";
    }
    if (!env) outFunction << "(";
    transitions(nodePtr->ptr1);
    if (!env) outFunction << ")";
  }
  //else treeError("default_case_opt");
}

/* 36) transitions :  logic_OR ';' | '{' logic_list '}' */
void Generate::transitions(TreeNode* nodePtr) {
  //if (mode == read_write__2) cerr << "write trans\n";
  err("trans\n");
  if (nodePtr->kind == transitions__1) {
    logic_OR(nodePtr->ptr1);
  }
  else {//if (nodePtr->kind == transitions__2) {
    logic_list(nodePtr->ptr1);
  }
  //else treeError("transitions");
}

/* 37) logic_list : logic_OR ';' | logic_OR ';' logic_list */
void Generate::logic_list(TreeNode* nodePtr) {
  if (nodePtr->kind == logic_list__1) {
    if (!env) outFunction << "(";
    logic_OR(nodePtr->ptr1);
    if (!env) outFunction << ")";
  }
  else { //if (nodePtr->kind == logic_list__2) {
    if (!env) { 
      outFunction << "(";
    }
    logic_OR(nodePtr->ptr1);
    if (!env) outFunction << ")";
    if (!env && optCnt%OPMAX == 0) outFunction << "\n   ";
    if (!env) { 
      outFunction << " && ";
    }
    logic_list(nodePtr->ptr2);
  }
  //else treeError("logic_list");
}

/* 38) logic_OR : logic_AND | logic_OR OROR logic_AND */
void Generate::logic_OR(TreeNode* nodePtr) {
  //if (mode == read_write__2) cerr << "  write logic_OR\n";

  if (nodePtr->kind == logic_OR__1) {
    logic_AND(nodePtr->ptr1);
  }
  else { //if (nodePtr->kind == logic_OR__2) {
    if (!env) {
      outFunction << "(";
    }
    logic_OR(nodePtr->ptr1);
    if (!env && optCnt%OPMAX == 0) outFunction << "\n    ";
    if (!env) {
      outFunction << " || ";
    }
    logic_AND(nodePtr->ptr2);
    if (!env) outFunction << ")";
  }
  //else treeError("logic_OR");
}

/* 39) logic_AND : logic_NOT | logic_AND ANDAND logic_NOT */
void Generate::logic_AND(TreeNode* nodePtr) {
  if (nodePtr->kind == logic_AND__1) {
    logic_NOT(nodePtr->ptr1);
  }
  else if (nodePtr->kind == logic_AND__2) {
    if (!env) {
      outFunction << "(";
    }
    logic_AND(nodePtr->ptr1);
    if (!env && optCnt%OPMAX == 0) outFunction << "\n    ";
    if (!env) {
      outFunction << " && ";
    }
    logic_NOT(nodePtr->ptr2);
    if (!env) outFunction << ")";
  }
  else treeError("logic_AND");
}

/* 40) logic_NOT : '!' logic_end | logic_end */
void Generate::logic_NOT(TreeNode* nodePtr) {
  if (nodePtr->kind == logic_NOT__1) {
    if (!env) {
      outFunction << " ! ";
    }
    logic_end(nodePtr->ptr1);
  }
  else if (nodePtr->kind == logic_NOT__2) {
    logic_end(nodePtr->ptr1);
  }
  else treeError("logic_NOT");
}

/* 41) logic_end : '(' logic_OR ')' | rel_exp */
void Generate::logic_end(TreeNode* nodePtr) {
  if (nodePtr->kind == logic_end__1) {
    logic_OR(nodePtr->ptr1);
  }
  else if (nodePtr->kind == logic_end__2) {
    rel_exp(nodePtr->ptr1);
  }
  else treeError("logic_end");
}

/* 42) rel_exp : add_exp relop add_exp */
void Generate::rel_exp(TreeNode* nodePtr) {
  err("rel\n");
  if (nodePtr->kind == rel_exp__1) {
    add_exp(nodePtr->ptr1);
    if (!env) relop(nodePtr->ptr2);
    add_exp(nodePtr->ptr3);
  }
  else treeError("rel_exp");
}

/* 43) relop :   '>' | GEQ | '<' | LEQ | EQEQ | NEQ */
void Generate::relop(TreeNode* nodePtr) {
  //if (mode == read_write__2) cerr << "write relop " << nodePtr->kind << endl;

  if (nodePtr->kind == relop__1) {
    outFunction << ">"; 
  }
  else if (nodePtr->kind == relop__2) {
    outFunction << ">="; 
  }
  else if (nodePtr->kind == relop__3) {
    outFunction << "<"; 
  }
  else if (nodePtr->kind == relop__4) {
    outFunction << "<="; 
  }
  else if (nodePtr->kind == relop__5) {
    outFunction << "=="; 
  }
  else if (nodePtr->kind == relop__6) {
    outFunction << "!="; 
  }
  else treeError("relop");
  optCnt++;
}

/* 44) add_exp : mul_exp | add_exp addop mul_exp ;*/
void Generate::add_exp(TreeNode* nodePtr) {
  err("add\n");
  if (nodePtr->kind == add_exp__1) {
    mul_exp(nodePtr->ptr1);
  }
  else if (nodePtr->kind == add_exp__2) {
    add_exp(nodePtr->ptr1);
    if (!env) addop(nodePtr->ptr2);
    mul_exp(nodePtr->ptr3);
  }
  else treeError("add_exp");
}

/* 45) addop : '+' | '-' */
void Generate::addop(TreeNode* nodePtr) {
  if (nodePtr->kind == addop__1) {
    outFunction << "+"; 
  }
  else if (nodePtr->kind == addop__2) {
    outFunction << "-"; 
  }
  else treeError("addop");
}

/* 46) mul_exp : exp_end | mul_exp '*' exp_end */
void Generate::mul_exp(TreeNode* nodePtr) {
  if (nodePtr->kind == mul_exp__1) {
    exp_end(nodePtr->ptr1);
  }
  else if (nodePtr->kind == mul_exp__2) {
    mul_exp(nodePtr->ptr1);
    if (!env) outFunction << "*"; 
    exp_end(nodePtr->ptr2);
  }
  else treeError("mul_exp");
}

/* 47) exp_end : '(' add_exp ')' | ID | num | BOOL | OLD '(' ID ')' | VALUE */
void Generate::exp_end(TreeNode* nodePtr) {
  if (nodePtr->kind == exp_end__1) {      //  '(' add_exp ')'
    if (!env) outFunction << "(";
    add_exp(nodePtr->ptr1);
    if (!env) outFunction << ")";
  }
  else if (nodePtr->kind == exp_end__2) { // ID
    id(nodePtr->ptr1, io);
    //cerr << nodePtr->ptr1->strval;
  }
  else if (nodePtr->kind == exp_end__3) { // num
    if (!env) outFunction << nodePtr->ptr1->ptr1->numval; 
    //cerr << nodePtr->ptr1->ptr1->numval; 
  }
  else if (nodePtr->kind == exp_end__4) { // bool
    if (!env) {
      if (nodePtr->ptr1->strval == "T")
	outFunction << "1";
      else 
	outFunction << "0";
    }
    //cerr << nodePtr->ptr1->strval;
  }
  else if (nodePtr->kind == exp_end__5) { // Old (ID)
    id(nodePtr->ptr1, "in");
    //cerr << nodePtr->ptr1->strval;
  }
  else if (nodePtr->kind == exp_end__6) { // VALUE
    if (!env) outFunction << "V";
    //cerr << "value";
  }
  else { // COUNT
    if (!env) outFunction << "C";
    //cerr << "count";
  }
}
 
void Generate::id(TreeNode* nodePtr, string ending) {
  SemanticRec * tmp = scope->lookup(nodePtr->strval);

  if (env) {
    if (tmp->kind == STATEINFO) { // state
      setUsedState(tmp->key);
      //cerr << "---  using " << tmp->key << " " << ending << endl;
      if (ending == "out") 
	setChangedState(tmp->key);
    }
  }

  else {
    if (tmp == NULL)                   // value
      outFunction << "V";

    else if (tmp->kind == FIELDINFO)   // field
      outFunction << getFieldLetter(tmp->key);
  
    else if (tmp->kind == STATEINFO)  // state
      outFunction << getStateLetter(tmp->key) + ending;
   
    else if (tmp->kind == ENUMINFO) // enum value
      outFunction << tmp->num;
  }
}

/******************** Generate an IO Clay read function for a port 
 * Needs state count, types, names, letters
 *       field count, sizes, names, letters
 *       mode (read_write__ 1/2/3)
 *       io ("in"/"out")
 */

// The states ", u32 E"...one for each used state
int Generate::createStateDecls(int n) {
  // The states ", u1 P, u32 Q, u2 R;"
  for(int i = 0 ; i<stateCount ; i++) {
    if ((changedStates.at(i) && io=="out") ||
	(usedStates.at(i) && io=="in")) {
      // print the separater
      if (n>0) outFunction << ", ";
      if (n%8 == 0 && n!=0) outFunction << "\n    ";
      
      // print the declaration
      outFunction << "u" << stateSizes.at(i)
		  << " " << stateLetters.at(i) << io;
      
      n++;
    }
  }
  return n;
}

// The defined bit ranges ", u1 F, u2 G"...one for each bit range
int Generate::createFieldDecls(int n) {
  SemanticRec* currBit;

  // The defined bit ranges ", u1 F, u2 G"...one for each bit range
  for(int i = 0 ; i<fieldCount ; i++) {

    // print the seperator
    if (n>0) outFunction << ", ";
    if (n%6 == 0 && n!=0) outFunction << "\n    ";

    // print the declaration
    outFunction << "u" << fieldSizes.at(i)+1 
		<< " " << fieldLetters.at(i); 
    n++;
  }
  return n;
}

// The overall state constraints " && (0<=G && G<=4)"
int Generate::printGlobalStateConstraints(int n) {
  SemanticRec* currRec;
  
  for(int i = 0 ; i<stateCount ; i++) {
    
    if((changedStates.at(i) && io=="out") ||
       (usedStates.at(i) && io=="in")) {
      // if it has any overall constraints
      currRec = scope->lookup(stateNames.at(i));
      if (!currRec->range.isEmpty()) { 
	
	// Print any connectors to previous constraints
	if (n%4 == 0)
	  outFunction << "\n    ";
	if (n>0) outFunction << " && "; // joining a set
	      
	// Print all constraints for this state
	n = printOneConstraints(currRec, stateLetters.at(i), n);	      
      }
    }
  }
  return n;
}

/* Prints constraints for one state or bit range to an IO Clay function
 * (A==3 && B==4 && (C==3 || C==2) && (2<=D && D<=6))
 * Prints to outFunction.
 */
int Generate::printOneConstraints(SemanticRec* currRec, string letter, int n) {
  Range currRange;
  outFunction << "(";

  // Print each part of the constraint
  //cerr << "range size is " << currRec->range.getSize() << endl;
  for(int i = 0 ; i < currRec->range.getSize() ; i++) {
    currRange = currRec->range.getRange(i);

    // || between possibilities "G==2 || G==4"
    if(i > 0)
      outFunction << " || ";

    // A value "G==3"
    if(currRange.getLow() == currRange.getHigh()) {
      outFunction << letter << io
		  << "==" << currRange.getLow();
      n++;
    }
    
    // A range "(0<=G && G<=4)"
    else { 
      outFunction << "(" << letter << io
		  << ">=" << currRange.getLow()
		  << " && " << letter << io
		  << "<=" << currRange.getHigh() <<")"; 
      n += 2;
    }
  }

  outFunction << ")";
  return n;
}

// The states ", windowState[D,G]"
int Generate::createStateTypes(int n) {
  bool changed;
 
  // The states ", windowState[D,G]"
  for(int i = 0 ; i<stateCount ; i++) {
    if (usedStates.at(i)) {
      // print the seperator
      outFunction << ", ";
      if (n%3 == 0)
	outFunction << "\n    ";
      n++;
	  
      // print the type
      if (changedStates.at(i)) 
	outFunction << getStateType(stateNames.at(i)) 
		    << "[D," << stateLetters.at(i) << "out]";
      else
	outFunction << getStateType(stateNames.at(i)) 
		    << "[D," << stateLetters.at(i) << "in]";
    }
  }
  return n;
}

// The fields ", Int[Eout], Int[Fout]"
int Generate::createFieldTypes(int n) {
  // In: the bit range variables ", Int[F] one, Int[G] two"
  // Out: the bit range types ", Int[E], Int[F]"
  for(int i = 0 ; i<fieldCount ; i++) {

    // print the seperator
    outFunction << ", ";
    if (n%7 == 0) outFunction << "\n    ";
      
    // print the type
    outFunction << "Int[" << fieldLetters.at(i) << "]";
    //if (io == "in")
    //    outFunction << " " << bitNames.at(i);
    n++;
  }
  return n;
}

//states ", windowState[D,X] window"
int Generate::createStateParams(int n){
  for(int i = 0 ; i<stateCount ; i++) {
    if (usedStates.at(i)) {
      // print the seperator
      outFunction << ", ";
      if (n%2 == 0) outFunction << "\n    ";
    
      // print the parameter
      outFunction << getStateType(stateNames.at(i)) 
		  << "[D," << stateLetters.at(i) << "in"
		  << "] " << getStateVar(stateNames.at(i));
    
      n++;
    }
  }
  return n;
}

//states ", Int[D,X] one"
int Generate::createFieldParams(int n){
  for(int i = 0 ; i<fieldCount ; i++) {
    // print the seperator
    outFunction << ", ";
    if (n%3 == 0 || (i==0 && n%3==2)) outFunction << "\n    ";
    
    // print the parameter
    outFunction << "Int[" << fieldLetters.at(i) << "] field_" 
		<< fieldNames.at(i);
    n++;
  }
  return n;
}

// "port14_Status_read_2byte" 
void Generate::createFunctionName(string rw, fstream & out) {
  out << rw << "_" << currPort->name;
}

// are any fields (besides this one) preserved?
bool Generate::otherPreserves(string field) {
  SemanticRec* nextRec;
  for(int i = 0 ; i < fieldCount ; i++)
    if (field != fieldNames.at(i)) {
      nextRec = scope->lookup(fieldNames.at(i));
      if (nextRec->write_attrib == write_attrib__2)
	return true;
    }

  return false;
}

void Generate::createEnumConstraints() {
  if (fieldCount == 0) return; // if no fields, there wont be any enums
  SemanticRec* nextRec;
  int consCount = 0;

  // find all enums and generate constraints
  scope->startIterator();
  while (scope->moreToIterate()) {
    nextRec = scope->nextIteratedSR();
    if (nextRec->kind == ENUMINFO) {
      if (optCnt > 0 && consCount == 0) outFunction << " && (";
      if (optCnt%4 == 0) outFunction << "\n    ";
      if (consCount != 0) outFunction << " || ";
      outFunction << getFieldLetter(nextRec->name) 
		  << "==" << nextRec->num;
      optCnt++; consCount++;
    }
  }
  // if opened parens (previous ops and enum ops), close the parens
  if (optCnt > consCount && consCount > 0)
    outFunction << ")";
}

/* 
 * A read function for a whole port 

 * native exists [declare fields or declare V if no fields, 
 *                declare changed states; 
 *     state constraints && postcondition && enum constraints]
 * @[baseState[D,A], state types, field types or Int[V]]
 * read_function
 * [u32 A, u32 D, declare states; state constraints && precondition]
 * (Int[A] addr, baseState[D,A] base, states);

 * For each field in a readable port

 * native exists [declare field (just one), 
 *                declare changed states; 
 *     state constraints && postcondition && enum constraints]
 * @[baseState[D,A], state types, field types or Int[V]]
 * field_read_function
 * [u32 A, u32 D, declare states; state constraints && precondition]
 * (Int[A] addr, baseState[D,A] base, states) {
 *    let [] (states, fields) = read_function(addr, base, states);
 *    return @(states, field);
 * }

 * A write function for a whole port

 * native exists [declare changed states; 
 *     state constraints && postcondition]
 * @[baseState[D,A], state types]
 * write_function
 * [u32 A, u32 D, declare states, declare fields or V;
 *  state constraints && precondition && enum constraints]
 * (Int[A] addr, baseState[D,A] base, states, fields);

 * For each field in a writeable port
 * (Only if also readable AND no preserved)

 * native exists [declare changed states; 
 *     state constraints && postcondition && enum constraints]
 * @[baseState[D,A], state types]
 * field_write_function
 * [u32 A, u32 D, declare states, declare field;
 *  state constraints && precondition where 
 *           fixed fields are replaced with fix value
 *           default fields are replaced with 0
 *           preserved fields???
 *  && enum constraints for that field && read precondition (if preserves)]
 * (Int[A] addr, baseState[D,A] base, states, field) {
 *    // if any fields preserved
 *    let [] (states, fields) = read_function(addr, base, states);
 *    return write_field(addr, base, states, preserved fields, 
 *                       fixed fields, 0 fields, that field);
 * }

 * The preserved fields cause a problem on the constraints of a field_write
 * Therefore, I still cannot implement them.
 */
void Generate::createIOReadFunction() {
  int n;
  mode = read_write__1; // read
  io = "out";

  //****** Existential part of the return type ******
  // "native exists [u32 V, u32 E, u1 F, u2 G; 
  //     (0<=G && G<=4) && E==1 && G==2] "

  // The basic returned value "native exists [u32 V"
  outFunction << "native exists [";

  // The states ", u32 E"...one for each used state
  n = createStateDecls(0);

  // The defined bit ranges ", u1 F, u2 G"...one for each bit range
  if (fieldCount>0) n = createFieldDecls(n);
  else {
    if (n>0) outFunction << ", ";
    outFunction << "u" << currPort->size*8 << " V";
  }

  outFunction << "; true"; 

  // The overall state constraints " && (0<=G && G<=4)"
  n = printGlobalStateConstraints(1);

  // The local state and field constraints " && E==1 && G==2"
  optCnt = n;
  after_opt(afterTree);

  // The enum constraints
  createEnumConstraints();

  // Finish the existentials "] "
  outFunction << "]\n";

  //****** Linear tuple part of the return type ******
  // "@[baseState[D,A], Int[V], windowState[D,G]], Int[E], Int[F]]"

  // The base state and whole returned value "@[baseState[D,A], Int[V]"
  outFunction << "   @[" << getStateType("Base") << "[D,A]";

  // The states ", windowState[D,Gout]"
  n = createStateTypes(1);

  // The fields ", Int[E], Int[F]"
  if (fieldCount>0) createFieldTypes(n*2);
  else outFunction << ", Int[V]";  

  // The end "]"
  outFunction << "]"<<endl;

  //****** function name ******
  createFunctionName("read", outFunction); 
  outFunction << "\n";
  
  io = "in";

  //****** declare type variables ******
  // "[u32 A, u32 D, u32 P, u1 E, u32 G; (0<=G && G<=4) && E==1 && G==2]"

  // The variables we know about "[u32 A, u32 D"
  outFunction << "   [u32 A, u32 D";

  // The states ", u1 P, u32 Q, u2 R;"
  createStateDecls(2);

  outFunction << "; true";

  // The overall state constraints " && (0<=G && G<=4)"
  n = printGlobalStateConstraints(1);

  // The local state constraints " && E==1 && G==2" (no field constraints)
  optCnt = n;
  before_opt(beforeTree);

  outFunction << "]\n";
  
  //****** parameters ******
  // "(Int[A] addr, baseState[D,A] base, 
  //   windowState[D,G] window);"
  outFunction << "   (Int[A] addr, " << getStateType("Base") << "[D,A] "
	      << getStateVar("Base");

  // states ", windowState[D,X] window"
  createStateParams(1);

  outFunction << ");\n\n";
}

/* A write function for the whole port */
void Generate::createIOWriteFunction() {
  int n = 0;
  int i;
  mode = read_write__2; // read
  io = "out";

  //****** Existential part of the return type ******
  // "native exists [u32 V, u32 E, u1 F, u2 G; 
  //     (0<=G && G<=4) && E==1 && G==2] "

  // The basic returned value "native exists ["
  outFunction << "native ";

  // *** currently drops post conditions if they only effect the value
  // *** those should have been pre-conditions
  // count number of changed states, if any, we need "exists"
  for (i=0; i<stateCount; i++)
    if (changedStates.at(i)) n++;
  if (n > 0) {
    outFunction << "exists [";

    // The states ", u32 E"...one for each used state
    n = createStateDecls(0);

    outFunction << "; true"; 

    // The overall state constraints " && (0<=G && G<=4)"
    n = printGlobalStateConstraints(1);

    // The local state and field constraints " && E==1 && G==2"
    optCnt = n;
    after_opt(afterTree);

    // Finish the existentials "] "
    outFunction << "]\n";
  }

  //****** Linear tuple part of the return type ******
  // "@[baseState[D,A], windowState[D,G]], Int[E], Int[F]]"

  // The base state and whole returned value "@[baseState[D,A], Int[V]"
  outFunction << "@[" << getStateType("Base") << "[D,A]";

  // The states ", windowState[D,Gout]"
  n = createStateTypes(1);

  // The end "]"
  outFunction << "]"<<endl;

  //****** function name ******
  createFunctionName("write", outFunction); 
  outFunction << "\n";
  
  io = "in";

  //****** declare type variables ******
  // "[u32 A, u32 D, u32 P, u1 E, u32 G; (0<=G && G<=4) && E==1 && G==2]"

  // The variables we know about "[u32 A, u32 D"
  outFunction << "   [u32 A, u32 D";

  // The states ", u1 P, u32 Q, u2 R;"
  createStateDecls(3);

  // The defined bit ranges ", u1 F, u2 G"...one for each bit range
  if (fieldCount>0) n = createFieldDecls(n);
  else outFunction << ", u" << currPort->size*8 << " V";

  outFunction << "; true";

  // The overall state constraints " && (0<=G && G<=4)"
  n = printGlobalStateConstraints(1);

  // The local state constraints " && E==1 && G==2" (no field constraints)
  optCnt = n;
  before_opt(beforeTree);

  // The enum constraints
  createEnumConstraints();

  outFunction << "]\n";
  
  //****** parameters ******
  // "(Int[A] addr, baseState[D,A] base, 
  //   windowState[D,G] window);"
  outFunction << "   (Int[A] addr, " << getStateType("Base") 
	      << "[D,A] " << getStateVar("Base");

  // states ", windowState[D,X] window"
  n = createStateParams(1);

  // fields ", Int[L] one"
  if (fieldCount>0) createFieldParams(n);
  else outFunction << ", Int[V] value";

  outFunction << ");\n\n";
}

void Generate::createIORepeatedFunction(int rw) {
  int n;
  mode = rw;
  io = "out";

  //****** Existential part of the return type ******
  // "native exists [u32 V, u32 E, u1 F, u2 G; 
  //     (0<=G && G<=4) && E==1 && G==2] "

  outFunction << "native ";

  // The basic returned value "native exists [u32 V"
  for (int i=0; i<stateCount; i++)
    if (changedStates.at(i)) n++;
  if (n > 0)
    outFunction << "exists [";

  // The states ", u32 E"...one for each used state
  n = createStateDecls(0);

  outFunction << "; true"; 

  // The overall state constraints " && (0<=G && G<=4)"
  n = printGlobalStateConstraints(1);
  
  // The local state and field constraints " && E==1 && G==2"
  optCnt = n;
  after_opt(afterTree);

  // Finish the existentials "] "
  outFunction << "]\n";

  //****** Linear tuple part of the return type ******
  // "@[baseState[D,A], Int[V], windowState[D,G]], Int[E], Int[F]]"

  // The base state and whole returned value "@[baseState[D,A], Int[V]"
  outFunction << "   @[" << getStateType("Base") << "[D,A]";

  // The states ", windowState[D,Gout]"
  n = createStateTypes(1);

  // The end "]"
  outFunction << "]"<<endl;

  //****** function name ******
  if (rw == read_write__1)
    createFunctionName("read", outFunction); 
  else
    createFunctionName("write", outFunction); 
  outFunction << "\n";
  
  io = "in";

  //****** declare type variables ******
  // "[u32 A, u32 D, u32 P, u1 E, u32 G; (0<=G && G<=4) && E==1 && G==2]"

  // The variables we know about "[u32 A, u32 D"
  outFunction << "   [u32 A, u32 D, u32 C, u32 B";

  // The states ", u1 P, u32 Q, u2 R;"
  createStateDecls(2);

  outFunction << "; true";

  // The overall state constraints " && (0<=G && G<=4)"
  n = printGlobalStateConstraints(1);

  // The local state constraints " && E==1 && G==2" (no field constraints)
  optCnt = n;
  before_opt(beforeTree);

  outFunction << "]\n";
  
  //****** parameters ******
  // "(Int[A] addr, baseState[D,A] base, 
  //   windowState[D,G] window);"
  outFunction << "   (Int[A] addr, " << getStateType("Base") << "[D,A] "
	      << getStateVar("Base")
	      << ", Int[C] count, Int[B] buffer";

  // states ", windowState[D,X] window"
  createStateParams(4);

  outFunction << ");\n\n";
}

/* Create read/write macro */
void Generate::createMacro(string io){
  int n = 1;

  // macro header
  outMacro << "\n#define ";
  if (io == "read") createFunctionName("R", outMacro);
  else createFunctionName("W", outMacro);
  outMacro << "(";

  // repeated ports
  if (currPort->type == 1)
    outMacro << "count, buffer";

  // fields or value
  else {
    if(fieldCount > 0)
      for(int i = 0 ; i < fieldCount ; i++) {
	if (i != 0) outMacro << ", ";
	outMacro << fieldNames.at(i);
      }
  
    else outMacro << "value";
  }

  // finish header
  outMacro << ") \\" << endl;
    
  // let these (read and write)
  outMacro << "let";

  // unpack?
  n = 0;

  // read not repeated (has values to return even if no states change)
  if (io == "read" && currPort->type != 1) outMacro << " []";

  // w/ changed states
  else {
    for (int i=0; i<stateCount; i++)
      if (changedStates.at(i)) n++;
    if (n > 0) outMacro << " []";
  }

  // base - never changes so safe to use Base2
  outMacro << " (" << getStateVar("Base") << "##2";

  // output states (read and write)
  for(int i = 0 ; i < stateCount ; i++)
    if (usedStates.at(i)) {
      outMacro << ", ";
      if (n%4 == 0) outMacro << "\\\n    ";
      if (changedStates.at(i))
	outMacro << getStateVar(stateNames.at(i));
      else
	outMacro << getStateVar(stateNames.at(i)) << "##2";
      n++;
    }

  // output fields, or value (read, not repeated)
  n = 1;
  if (io == "read" && currPort->type != 1) {
    if(fieldCount > 0)
      for(int i = 0 ; i < fieldCount ; i++) {
	outMacro << ", ";
	outMacro << fieldNames.at(i);
	n++;
      }
    
    else {
      outMacro << ", value";
      n++;
    }
  }

  // finish the let (read and write)
  outMacro << ") = \\" << endl;
  
  // equal this function (read and write)
  outMacro << "  ";
  createFunctionName(io, outMacro);
  outMacro << "(IOADDR, " << getStateVar("Base");
  n = 2;

  // repeated ports
  if (currPort->type == 1)
    outMacro << ", count, buffer";

  // input states (read and write)
  for(int i = 0 ; i < stateCount ; i++)
    if (usedStates.at(i)) {
      outMacro << ", ";
      if (n%4 == 0) outMacro << "\\\n    ";
      outMacro << getStateVar(stateNames.at(i));
      n++;
    }

  // input arguments (write, not repeated)
  if (io == "write" && currPort->type != 1) {
    if(fieldCount > 0)
      for(int i = 0 ; i < fieldCount ; i++) {
	outMacro << ", ";
	outMacro << fieldNames.at(i);
	n++;
      }

    else {
      outMacro << ", value";
      n++;
    }
  }

  // end of input (read and write)
  outMacro << "); \\" << endl;
  
  // link states (read and write)
  for(int i = 0 ; i < stateCount ; i++)
    if (usedStates.at(i) && ! changedStates.at(i))
      outMacro << "  " << getStateVar(stateNames.at(i)) << " = " 
	       << getStateVar(stateNames.at(i)) << "##2; \\\n  ";
  outMacro << getStateVar("Base") << " = " << getStateVar("Base") 
	   << "##2;\n\n";
}

/* Create native read function */
void Generate::createNativeReadFunction(){
  SemanticRec* currBit;
    
  // repeated ports
  if (currPort->type == 1) {
    //****** Header ******
    outNative << "  inline void "; 
    createFunctionName("read", outNative);

    //****** Parameters ******
    outNative << "(unsigned long addr, unsigned long count"
	      << ", unsigned long buffer) {\n";

    //****** Do the read ******
    outNative << "    " << ioFunctionName("read", currPort->size, true)
	      << "(addr + " << currPort->num << ", count, buffer);\n";

    //****** Return ******
    outNative << "    return;\n";
    outNative << "  } \n\n" << endl;
  }
  else {
    //****** Header ******
    // "  inline unsigned long port14_Status_default_read_2byte"
    outNative << "  inline "; 
    
    // return type "unsigned long" or "struct Clay_Obj<3>"
    if(fieldCount < 2)
      outNative << "unsigned long "; 
    else //bit ranges need to be taken out
      outNative << "struct Clay_Obj<" << fieldCount << "> ";
    
    createFunctionName("read", outNative);

    //****** Parameters ******
    outNative << "(unsigned long addr) {\n";

    //****** Do the read ******
    outNative << "    int value = " 
	      << ioFunctionName("read", currPort->size, false)
	      << "(addr + " << currPort->num << ");\n";

    //****** Assemble return value ******
    if(fieldCount > 1) { //bit ranges exist
      int start;
      int finish;

      outNative << "    struct Clay_Obj<" << fieldCount << "> c;\n";

      for(int i = 0 ; i<fieldCount ; i++){
	currBit = scope->lookup(fieldNames.at(i));

	start = currBit->range.getRange(0).getLow();
	finish = currBit->range.getRange(0).getHigh();

	outNative << "    c.A[" << i << "] = (value >> " 
		  << start << ") & " << genReadMask(start, finish)
		  << ";" << endl;
      }
    }
    else if (fieldCount == 1) { // 1 range (might be less than all bits)
      int start;
      int finish;
      currBit = scope->lookup(fieldNames.at(0));
      
      start = currBit->range.getRange(0).getLow();
      finish = currBit->range.getRange(0).getHigh();
      outNative << "    value = (value >> " 
		<< start << ") & " << genReadMask(start, finish)
		<< ";" << endl;
    }

    //****** Return it ******
    outNative << "    return ";

    if(fieldCount > 1) // struct Clay_Obj
      outNative << "c;" << endl;
    else             // unsigned long
      outNative << "value;" << endl;
    outNative << "  } \n\n" << endl;
  }

}

/* Create native write function */
void Generate::createNativeWriteFunction(){
  SemanticRec* currBit;
  int start, finish;
  
  // repeated ports
  if (currPort->type == 1) {
    //****** Header ******
    outNative << "  inline void "; 
    createFunctionName("write", outNative);

    //****** Parameters ******
    outNative << "(unsigned long addr, unsigned long count"
	      << ", unsigned long buffer) {\n";

    //****** Do the read ******
    outNative << "    " << ioFunctionName("write", currPort->size, true)
	      << "(addr + " << currPort->num << ", count, buffer);\n";

    //****** Return ******
    outNative << "    return;\n";
    outNative << "  } \n\n" << endl;
  }
  else {
    //****** Header ******
    // "  inline void port14_Status_write_2byte("
    outNative << "  inline void ";

    createFunctionName("write", outNative);

    //****** Parameters ******
    outNative << "(unsigned long addr";

    if(fieldCount > 0) { //bit ranges exist
      int start = 0;
      int finish = 0;
	
      for(int i = 0 ; i<fieldCount ; i++) {
	if (i%3 == 0) outNative << ",\n  ";
	else          outNative << ", ";
	outNative << "unsigned long " << fieldNames.at(i);
      }
      outNative << ") {\n";
    }
    else {
      outNative << ", unsigned long value) {\n";
    }

    //****** If any reserved fields are also preserved, call read *****
    //****** WARNING : DOESNT CHECK THAT A READ CALL IS ALLOWED
    //****** WARNING : THIS MAY CAUSE AN EXTRA READ CALL IF OTHERS PRESERVED

    //****** Assemble write value ******
    if(fieldCount > 0) { //bit ranges exist
      for(int i = 0 ; i<fieldCount ; i++) {
	currBit = scope->lookup(fieldNames.at(i));

	start = currBit->range.getRange(0).getLow();
	finish = currBit->range.getRange(0).getHigh();

	outNative << "    " << fieldNames.at(i) << " = (" << fieldNames.at(i) 
		  << " << " << start << ") & " 
		  << genWriteMask(start, finish) << ";" << endl;
      }
    }
	
    //****** Do the write ******
    outNative << "    " << ioFunctionName("write", currPort->size, false)
	      <<"(addr + " << currPort->num << ", ";

    if(fieldCount > 0) {
      for(int i = 0 ; i<fieldCount-1 ; i++)
	outNative << fieldNames.at(i) << " + ";
      outNative << fieldNames.at(fieldCount-1) << "); \n";
    }
    else {
      outNative << "value); \n";
    }

    outNative << "  } \n\n" << endl;
  }
}



/**************** Helper Functions ************************/

int Generate::getFieldSize(string key) {
  int i;
  for (i=0; i<fieldCount && fieldNames.at(i) != key; i++);
  return fieldSizes.at(i);
}
int Generate::getStateSize(string key) {
  int i;
  for (i=0; i<stateCount && stateNames.at(i) != key; i++);
  return stateSizes.at(i);
}

string Generate::getFieldLetter(string key) {
  int i;
  for (i=0; i<fieldCount && fieldNames.at(i) != key; i++);
  return fieldLetters.at(i);
}
string Generate::getStateLetter(string key) {
  int i;
  for (i=0; i<stateCount && stateNames.at(i) != key; i++);
  return stateLetters.at(i);
}
void Generate::setUsedState(string key) {
  int i;
  for (i=0; i<stateCount && stateNames.at(i) != key; i++);
  usedStates.at(i) = true;
}
void Generate::setChangedState(string key) {
  int i;
  for (i=0; i<stateCount && stateNames.at(i) != key; i++);
  changedStates.at(i) = true;
}

string Generate::getStateType(string state) {
  return "StateType_" + state;
}
string Generate::getStateVar(string state) {
  return "stateVar_" + state;
}

string Generate::ioFunctionName(string rw, int size, bool repeated) {
  string str = rw + itoa(size) + "byte";
  if (repeated) {
    str = str + "rep";
    addIOCommand("void " + str + "(unsigned long addr, unsigned long count, unsigned long buffer)");
  }
  else if (rw == "read") {
    addIOCommand("unsigned long " + str + "(unsigned long addr)");
  }
  else 
    addIOCommand("void " + str + "(unsigned long addr, unsigned long value)");
  return str;
}

/* Add io command info */
void Generate::addIOCommand(string s){
  SemanticRec* tempRec = IOCommands->lookup(s);
  if(tempRec == NULL){
    tempRec = new SemanticRec(-1,-1);
    tempRec->name = s;
    IOCommands->insert(s, tempRec);
  }
}
