/* 
 * Laddie to Clay Compiler
 * Author: Lea Wittie and Derrin Pierret
 * Copyright (c) 2007 Bucknell University
 *
 * Permission is hereby granted, free of charge, to any individual or
 * institution obtaining a copy of this software and associated
 * documentation files (the "Software"), to use, copy, modify, and
 * distribute without restriction, provided that this copyright and
 * permission notice is maintained, intact, in all copies and supporting
 * documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL BUCKNELL UNIVERSITY BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/*
 * Static Semantics of Laddie
 * 
 * State names are global and unique
 * Port rule names are global and unique
 * Field names are unique within their port rule
 * no fields in the pre condition of a read
 * Old only in post conditions
 * Old only on states
 * No overlapping fields
 * Reserved fields may be fixed but not preserved** not impl here or 
 * in generate
 */

using namespace std;
#include <iostream>
#include <string>
#include "compiler.h"
#include "support.h"
#include "Semantics.h"
#include "morekinds.h"
#include <cmath>

//#define DEBUG
#ifdef DEBUG
#define err(s) cout << s ;
#else 
#define err(s)
#endif

#define DEBUG_declare(s)
//#define DEBUG_declare(s) cerr << "Declaring " << s << endl;

Semantics::Semantics() {
  scope = new SymbolTable();         // top level scope
  io = "";
  mode = 0;
  errCount = 0;
}

void Semantics::walkTree(TreeNode* nodePtr) {
  err("walkTree\n");
  driver(nodePtr);
  if (errCount > 0) exit(1);
}

/* 1) driver : states_opt ports */
void Semantics::driver(TreeNode* nodePtr) {
  err("driver\n");
  states_opt(nodePtr->ptr1);
  ports(nodePtr->ptr2);
  nodePtr->symTabPtr = scope;   // save scope
}

/**********************global state declarations *****************************
 * name is the state name
 * info->range is the allowed values it can have
 * kind is from the type (int/boolean)
 * line is the line it was declared on
 */

/* 18) states_opt : | states */
void Semantics::states_opt(TreeNode* nodePtr) {
  if (nodePtr->kind == states_opt__2)
    states(nodePtr->ptr1);
}

/* 19) states : state | state states */
void Semantics::states(TreeNode* nodePtr) {
  state(nodePtr->ptr1);
  if (nodePtr->kind == states__2)
    states(nodePtr->ptr2);
}

/* 20) state : type ID ; | type ID colon_opt num_or_set ; 
 * Declaring linear singleton system state
 * - may not have two states with the same name
 * - integer states may constrain the allowed values (not boolean)
 * Make a Semantic Record (Field)
 */
void Semantics::state(TreeNode* nodePtr) {
  err("userdeffield "); err(nodePtr->ptr2->strval);err("\n");
  SemanticRec* currRec = scope->lookup(nodePtr->ptr2->strval);

  // ERROR if this state already exists
  if (currRec != NULL) {
    cerr << "Line " << nodePtr->ptr2->line << ": " 
	 << nodePtr->ptr2->strval << " is already defined on line " 
	 << currRec->line << endl;
    errCount++;
  }

  // Make a semantic record for the new state
  currRec = new SemanticRec(STATEINFO, nodePtr->ptr2->line);
  currRec->type = nodePtr->ptr1->kind;// integer or boolean
  currRec->key = nodePtr->ptr2->strval;
  scope->insert(currRec);

  // if it has special constraints
  if (nodePtr->kind == state__2) {
      
    // ERROR if it is a boolean "boolean foo allows [1-4]"
    if(nodePtr->ptr1->kind == type__1){
      cerr << "Line " << nodePtr->ptr2->line << ": " 
	   << nodePtr->ptr2->strval 
	   << " is improperly allowing a range" << endl;
      errCount++;
    }

    // setup its constraints
    num_or_set(nodePtr->ptr4, currRec);
  }
  //cerr << "range size is " << currRec->range.getSize() << endl;

}

/*************** general things ****************************/
/* dont need a function for num, value, type, num_or_range */

/* num_or_set : num | '[' constraint_set ']' 
 * Case 1: add the number to the given range
 * Used by StateInfos and FieldInfos
 */
void Semantics::num_or_set(TreeNode* nodePtr, SemanticRec* currRec) {
  // add the number to the overall range
  if (nodePtr->kind == num_or_set__1) {
    currRec->range.add(nodePtr->ptr1->ptr1->numval);
    //cerr << "adding " << nodePtr->ptr1->ptr1->numval << endl;
  }
  else
    constraint_set(nodePtr->ptr1, currRec);
}

/* constraint_set :       num_or_range
   | num_or_range ',' constraint_set
   * Case number: value possible in given number of bits
   * Case range: both ends possible in given number of bits
   */
void Semantics::constraint_set(TreeNode* nodePtr, SemanticRec* currRec) {

  // It's  number
  if(nodePtr->ptr1->kind == num_or_range__1) {

    // ERROR if a number value isnt possible in this number of bits
    if(currRec->kind == field_info__1
       && nodePtr->ptr1->ptr1->ptr1->numval > maxBitValue(currRec)){
      cerr << "Line " << nodePtr->ptr1->line << ": " 
	   << currRec->name 
	   << " is allowing a value that is not possible in its bit range" 
	   << endl;
      errCount++;
    }
	
    // add the number to the overall range
    currRec->range.add(nodePtr->ptr1->ptr1->ptr1->numval);
    //cerr << "adding " << nodePtr->ptr1->ptr1->ptr1->numval << endl;
  }

  // it's a range
  else {
	
    // ERROR if the range isnt possible in this number of bits
    if(currRec->kind == field_info__1
       && (nodePtr->ptr1->ptr1->ptr1->numval > maxBitValue(currRec)
	   || nodePtr->ptr1->ptr2->ptr1->numval > maxBitValue(currRec))){
      cerr << "Line " << nodePtr->ptr1->line << ": " 
	   << currRec->name
	   << "'s has an allowed value that is not possible in its bit range" 
	   << endl;
      errCount++;
    }

    // add the range to the overall range
    currRec->range.add(nodePtr->ptr1->ptr1->ptr1->numval,
		       nodePtr->ptr1->ptr2->ptr1->numval);
    //cerr << "adding " << nodePtr->ptr1->ptr1->ptr1->numval
    //	 << " to " << nodePtr->ptr1->ptr2->ptr1->numval << endl;

  }

  // go on the the next constraint
  if (nodePtr->kind == constraint_set__2)
    constraint_set(nodePtr->ptr2, currRec);
}

/****************************** ports ************************************
 * Get general info on the port
 *
 *
 */
/* 2) ports : port ports | port */
void Semantics::ports(TreeNode* nodePtr) {
  port(nodePtr->ptr1);
  if (nodePtr->kind == ports__1)
    ports(nodePtr->ptr2);
}

/* 3) port : PORT num '{' port_info and other info '}' */
void Semantics::port(TreeNode* nodePtr) {
  err("port "); err(nodePtr->ptr1->ptr1->numval);
  DEBUG_declare(nodePtr->ptr1->ptr1->numval);

  usedSwitchValues = new Boundary(); // collection of switch values
  reservedCount = 0; // number of reserved fields (named __reserved__#)

  port_info(nodePtr->ptr2, nodePtr->ptr1->ptr1->numval);

  nodePtr->strval = currPort->key; // save key for later
  nodePtr->symTabPtr = scope;   // save scope
  popScope(); 	// port is done, so kill port's scope
  
  /* NOT AN ERROR: multiple windows may have the same register 
     if (!isPortUnique()) {
     cerr << "Line " << currPort->line << ": Port number " 
     << currPort->num << " does not have a unique access mode." << endl;
     errCount++;

     }
  */
  delete usedSwitchValues;

}

/* 5) port_info : name_info size_info_opt type_info_opt access_info_opt 
   field_list_opt before_opt after_opt test_opt */
void Semantics::port_info(TreeNode* nodePtr, int num) {
  
  currPort = new SemanticRec(PORTINFO, nodePtr->ptr1->line);

  // Handle the port setup
  currPort->num = num;            // set number
  name_info(nodePtr->ptr1);       // verify name ok, set name, set key
  size_info_opt(nodePtr->ptr2);   // set portSize
  type_info_opt(nodePtr->ptr3);	  // set type
  access_info_opt(nodePtr->ptr4); // set access 

  // insert port into global scope
  scope->insert(currPort);
    
  // create local scope for port fields
  pushScope();
    
  // Fields in the port
  field_list_opt(nodePtr->ptr5);

  // Before constraints
  before_opt(nodePtr->ptr6);  
  delete usedSwitchValues; 
  usedSwitchValues = new Boundary();  
  usedSwitchNames.clear();

  // After constraints
  after_opt(nodePtr->ptr7);  
  delete usedSwitchValues; 
  usedSwitchValues = new Boundary();  
  usedSwitchNames.clear();

  // consistency testing
  if (consistency) {
    test_opt(nodePtr->ptr8);
  }
}

// if consistency is turned on and this port has test inputs, gather them.
// test_opt       : | TEST_INPUT test_list ';' ;
// test_list      : test | test_list ',' test ;
// test           : ID '=' num_or_bool ;
void Semantics::test_opt(TreeNode * nodePtr) {
  if (nodePtr->kind == test_opt__1) 
    currPort->length = -1;
  else {
    currPort->length = 0;
    test_list(nodePtr->ptr1);
  }
}
void Semantics::test_list(TreeNode * nodePtr) {
  if (nodePtr->kind == test_list__1)
    test(nodePtr->ptr1);
  else {
    test_list(nodePtr->ptr1);
    test(nodePtr->ptr2);
  } 
}
void Semantics::test(TreeNode * nodePtr) {
  if (nodePtr->ptr1->kind == id_or_value__1) { // ID
    SemanticRec* temp = scope->lookup(nodePtr->ptr1->ptr1->strval);

    // ERROR if ID not in scope
    if (temp == NULL) {
      cerr << "Line " << nodePtr->ptr1->ptr1->line << ": " 
	   << nodePtr->ptr1->ptr1->strval << " is not in scope." << endl;
      errCount++;
    }

    // ERROR if ID is not a state or field
    if (temp->kind != STATEINFO && temp->kind != FIELDINFO) {
      cerr << "Line " << nodePtr->ptr1->ptr1->line << ": " 
	   << nodePtr->ptr1->ptr1->strval << " is not a state or field." 
	   << endl;
      errCount++;
    }
  }

  // Convert booleans to 0 or 1, no semantic checking.
  currPort->length++;
  currPort->ids.resize(currPort->length);
  currPort->values.resize(currPort->length);
  if (nodePtr->ptr1->kind == id_or_value__1)
    (currPort->ids).at(currPort->length-1) = nodePtr->ptr1->ptr1->strval;
  else
    (currPort->ids).at(currPort->length-1) = "value";
  if (nodePtr->ptr2->kind == num_or_bool__2) {
    if (nodePtr->ptr2->ptr1->strval == "T") 
      (currPort->values).at(currPort->length-1) = 1;
    else (currPort->values).at(currPort->length-1) = 0;
  }
  else {
    (currPort->values).at(currPort->length-1) = 
      nodePtr->ptr2->ptr1->ptr1->numval;
  }
}

/* 6) name_info : NAME '=' ID ';' */
void Semantics::name_info(TreeNode* nodePtr) { //vector<string> & names) {
  //ERROR does port already exist?
  if(nameExists(nodePtr->ptr1->strval, usedPortNames)){
    cerr << "Line " << nodePtr->ptr1->line << ": the name " 
	 << nodePtr->ptr1->strval << " is already used." << endl;
    errCount++;
  }

  usedPortNames.push_back(nodePtr->ptr1->strval);
  currPort->name = nodePtr->ptr1->strval;
  currPort->key = currPort->name;
  err(nodePtr->ptr1->strval); err("\n");
}

/* 7) size_info : | SIZE '=' num ';' */
void Semantics::size_info_opt(TreeNode* nodePtr) {
  if (nodePtr->kind == size_info_opt__2)
    currPort->size = nodePtr->ptr1->ptr1->numval;
  else 
    currPort->size = 1; // default is 1 byte
}

// type_info : | TYPE = REPEATED
void Semantics::type_info_opt(TreeNode* nodePtr) {
  if (nodePtr->kind == type_info_opt__2)
    currPort->type = 1; // repeated
  else 
    currPort->type = 0; // not
}
/* 8) access_info : | ACCESS '=' read_write */
void Semantics::access_info_opt(TreeNode* nodePtr) {
  err("  access \n");
  if (nodePtr->kind == access_info_opt__1) {
    currPort->access = read_write__3;
  }

  else {
    currPort->access = nodePtr->ptr1->kind;
  }

}

/******************* fields in a port *****************************/

/* 14) field_list_opt : | field_list */
void Semantics::field_list_opt(TreeNode* nodePtr) {
  if (nodePtr->kind == field_list_opt__2) {
    field_list(nodePtr->ptr1);    
  }
}
/* 15) field_list : field_info | field_info field_list */
void Semantics::field_list(TreeNode* nodePtr) {
  field_info(nodePtr->ptr1);
  if (nodePtr->kind == field_list__2)
    field_list(nodePtr->ptr2);
}

/* 16) field_info : '[' num_or_range ']' colon_opt ID ';' field_attribs
 * The bit range needs to be 
 *   - not named the same as the the other bit ranges
 *   - possible in this port
 *   - not overlapping another bit range in this port
 * Multiple ranges may be named "reserved". we'll differentiate using __#
 */
void Semantics::field_info(TreeNode* nodePtr) {
  err("  field info "); err(nodePtr->ptr3->strval); err("\n");
  DEBUG_declare(nodePtr->ptr3->strval);

  // lookup the bit field's name (ID)
  currField = scope->lookup(nodePtr->ptr3->strval);
  
  // ERROR if field already exists
  if (currField != NULL) {
    cerr << "Line " << nodePtr->ptr3->line << ": " 
	 << nodePtr->ptr3->strval << " is already defined on line " 
	 << currField->line << endl;
    errCount++;
  }

  // Get the range ordered low, high
  int first = nodePtr->ptr1->ptr1->ptr1->numval;
  int second;
  int tmp;
  if (nodePtr->ptr1->kind == num_or_range__1) // only 1 bit
    second = first;
  else
    second = nodePtr->ptr1->ptr2->ptr1->numval;

  if (first > second) {
    tmp = second;
    second = first;
    first = tmp;
  }

  // ERROR if range is not possible in this port
  if (first >= currPort->size*8 || second >= currPort->size*8) {
    cerr << "Line " << nodePtr->ptr3->line << ": " <<nodePtr->ptr3->strval 
	 << "'s bits go outside the range of the port size" << endl;
    errCount++;
  }

  // Make a semantic record for the named bit range 
  currField = new SemanticRec(FIELDINFO, nodePtr->ptr3->line);
  if (nodePtr->ptr3->strval == "reserved") // set reserved key 
    currField->key = "__" + nodePtr->ptr3->strval 
      + "__" + itoa(reservedCount++);
  else
    currField->key = nodePtr->ptr3->strval; // set normal key 
  //newRec->type = type__2; // essentially an integer 
  currField->range.add(first, second); 

  // ERROR if any bits are shared between IDs
  scope->startIterator();
  SemanticRec* nextRec;
  while (scope->moreToIterate()) {
    nextRec = scope->nextIteratedSR();
    if(nextRec->kind == FIELDINFO && doBitsConflict(currField, nextRec) ){
      cerr << "Line " << nodePtr->ptr3->line << ": " << nodePtr->ptr3->strval 
	   << "'s bits are already assigned to " << nextRec->name << endl;
      errCount++;
    }
  }

  scope->insert(currField);

  field_attribs(nodePtr->ptr4);
}

/* field_attribs : read_attrib_opt write_attrib_opt enum_opt ; */
void Semantics::field_attribs(TreeNode* nodePtr) {
  read_attrib_opt(nodePtr->ptr1);
  write_attrib_opt(nodePtr->ptr2);
  enum_opt(nodePtr->ptr3);
}

void Semantics::read_attrib_opt(TreeNode* nodePtr) {
  // NOT IMPELMENETD
}

/* write_attrib_opt : | WRITE_VALUE '=' write_attrib ';' ; 
 * If we are specifically writing some other field, this field should be
 *   fixed(n): set to n
 *   preserved: read and use that value
 *   set to 0
 */
void Semantics::write_attrib_opt(TreeNode* nodePtr) {
  if (nodePtr->kind == write_attrib_opt__2)
    write_attrib(nodePtr->ptr1);
  else
    currField->write_attrib = 0;
}

/* write_attrib : FIXED '(' num ')' | PRESERVE 
 * default is none 
 * Error if the num won't fit in the field
 */
void Semantics::write_attrib(TreeNode* nodePtr) {
  currField->write_attrib = nodePtr->kind;

  if (nodePtr->kind == write_attrib__1) {

    int num = nodePtr->ptr1->ptr1->numval;

    // ERROR if num wont fit in field
    if (pow(2.0,currField->range.getRange(0).getHigh() - 
	    currField->range.getRange(0).getLow()) < num) {
      cerr << "Line " << nodePtr->ptr1->ptr1->line << ": " << num 
	   << "cannot fit in the bits of " << currField->key << endl;
      errCount++;
    }
  
    currField->write_num = num;
  }
}

// enum_opt                : | ENUM '{' enum_list '}' ';' ;
void Semantics::enum_opt(TreeNode* nodePtr) {
  if (nodePtr->kind == enum_opt__2) {
    enum_value = 0;
    enum_list(nodePtr->ptr1); 
  }
}
// enum_list               : enum_val | enum_val ',' enum_list ;
void Semantics::enum_list(TreeNode* nodePtr) {
  enum_val(nodePtr->ptr1);
  if (nodePtr->kind == enum_list__2)
    enum_list(nodePtr->ptr2);
}
// enum_val                : ID | ID '=' num ;
void Semantics::enum_val(TreeNode* nodePtr) {
  DEBUG_declare(nodePtr->ptr1->strval);
  SemanticRec* tempRec;

  // ERROR if already in scope (reguardless of its type)
  tempRec = scope->lookup(nodePtr->ptr1->strval);
  if (tempRec != NULL) {
    cerr << "Line " << nodePtr->ptr1->line << ": " 
	 << nodePtr->ptr1->strval << " is already defined on line " 
	 << tempRec->line << endl;
    errCount++;
  }
  
  // Put in scope so usable by conditions
  tempRec = new SemanticRec(ENUMINFO, nodePtr->ptr1->line);
  tempRec->key = nodePtr->ptr1->strval; // enum value name
  
  if (nodePtr->kind == enum_val__2) {
    tempRec->num = nodePtr->ptr2->ptr1->numval; // enum value value
    enum_value = tempRec->num + 1; // next value
  }
  else {
    tempRec->num = enum_value;
    enum_value++;
  }
  tempRec->name = "" + currField->key; // field name
  scope->insert(tempRec);
  
}
/********* constraints on before and after ************************/
/* Warning: Does not check for many inconsistencies in the spec.
 * Clay will catch those when it compiles the Clay code.
 * There can be multiple befores and afters. 
 * All of the befores must be before all of the afters (may change)
 */

/* before_opt       : | before before_opt */
void Semantics::before_opt(TreeNode* nodePtr) {
  io = "before";
  err("  before_opt\n");
  if (nodePtr->kind == before_opt__2) {
    before(nodePtr->ptr1);
    before_opt(nodePtr->ptr2);
  }
}

/* 22) before : BEFORE constraints | BEFORE read_write constraints */
void Semantics::before(TreeNode* nodePtr) {
  if (nodePtr->kind == before__1 || nodePtr->kind == before__3) { 
    mode = read_write__3;
    constraints(nodePtr->ptr1);
  }
  else {
    mode = nodePtr->ptr1->kind;
    constraints(nodePtr->ptr2);
  }
}

/* 22) after_opt : | after after_opt */
void Semantics::after_opt(TreeNode* nodePtr) {
  io = "after";
  err("  after_opt\n");
  if (nodePtr->kind == after_opt__2) {
    after(nodePtr->ptr1);
    after_opt(nodePtr->ptr2);
  }
}

/* after: AFTER constraints| AFTER RW constraints */
void Semantics::after(TreeNode* nodePtr) {
  if (nodePtr->kind == after__1 || nodePtr->kind == after__3) {
    err("1 or 3\n");
    mode = read_write__3;
    constraints(nodePtr->ptr1);
  }
  else {
    err("2 or 4\n");
    mode = nodePtr->ptr1->kind;
    constraints(nodePtr->ptr2);
  }
  err("  after ends\n");
}

/* 22) constraints : transitions 
 *      | SWITCH ID '{' switch_cases default_case_opt '}' 
 * The ID must be in scope 
 * and it must be a Field name
 */
void Semantics::constraints(TreeNode* nodePtr) {
  //if (nodePtr == NULL) cerr << "null node ptr\n";
  err("constraints\n"); 
  if (nodePtr->kind == constraints__1) {
    err("  cons 1\n");
    transitions(nodePtr->ptr1);
  }
  else {
    currSwitch = scope->lookup(nodePtr->ptr1->strval);

    // ERROR if this is a pre-condition on a read in a readable port
    if (io=="before" && mode!=read_write__3 
	&& currPort->access != read_write__3) {
      cerr << "Line " << nodePtr->ptr1->line << ": " 
	   << "Preconditions on reads cannot refer to fields." 
	   << endl;
      errCount++;
    }

    // ERROR if ID is not already declared
    if(currSwitch == NULL){
      cerr << "Line " << nodePtr->ptr1->line << ": " 
	   << nodePtr->ptr1->strval 
	   << " is undefined" 
	   << endl;
      errCount++;
    }

    // ERROR if ID is not a field
    if(currSwitch->kind != FIELDINFO){
      cerr << "Line " << nodePtr->ptr1->line 
	   << ": Switch cannot be applied to " 
	   << nodePtr->ptr1->strval 
	   << " because it is not a port field" 
	   << endl;
      errCount++;
    }

    switch_cases(nodePtr->ptr2);
    default_case_opt(nodePtr->ptr3);
  }
}

/* 24) switch_cases : switch_case | switch_case switch_cases */
void Semantics::switch_cases(TreeNode* nodePtr) {
  switch_case(nodePtr->ptr1);
  if (nodePtr->kind == switch_cases__2) {
    switch_cases(nodePtr->ptr2);
  }
}

/* 26) switch_case : num ':' transitions | id : trans
 * The num must be allowed in the switched ID's bit range
 * It must not have already been used in this switch
 */
void Semantics::switch_case(TreeNode* nodePtr) {
  int num;
  int line;

  if (nodePtr->kind == switch_case__1) {
    num = nodePtr->ptr1->ptr1->numval;
    line = nodePtr->ptr1->ptr1->line;
  }
  else {
    SemanticRec* tmp = scope->lookup(nodePtr->ptr1->strval);
    // ERROR if value not a declared enum
    if (tmp==NULL || tmp->kind != ENUMINFO) {
      cerr << "Line " << nodePtr->ptr1->line << ": " 
	   << nodePtr->ptr1->strval  
	   << " is not a declared enum value." 
	   << endl;
      errCount++;
      exit(1);
    }

    // ERROR if value not for this switch field
    if (tmp->name != currSwitch->key) {
      cerr << "Line " << nodePtr->ptr1->line << ": " 
	   << nodePtr->ptr1->strval  
	   << " is an enum value for " << tmp->name
	   << " not for " << currSwitch->key << "." 
	   << endl;
      errCount++;
    }

    num = tmp->num;
    line = nodePtr->ptr1->line;
  }

  // ERROR if the case value is not allowed in the ID's bit range
  if(num > maxBitValue(currSwitch)){
    cerr << "Line " << nodePtr->ptr1->ptr1->line << ": " 
	 << currSwitch->name 
	 << " has a switch case, " << num 
	 << ", that is not possible in its bit range." 
	 << endl;
    errCount++;
  }

  // ERROR if the num has already been used in this switch?
  if(!usedSwitchValues->isEmpty() && 
     usedSwitchValues->isInBounds(num)){
    cerr << "Line " << nodePtr->ptr1->ptr1->line << ": " 
	 << "Switch value "<< num 
	 << " has already been used as a switch case." 
	 << endl;
    errCount++;
  }

  // Add this num to the range of the switch
  usedSwitchValues->add(num);

  transitions(nodePtr->ptr2);
}

/* 27) default_case_opt : | DEFAULT : transitions 
 * If there is no default case, then only those values will be allowed (evetually)
 */
void Semantics::default_case_opt(TreeNode* nodePtr) {
  if (nodePtr->kind == default_case_opt__2)
    transitions(nodePtr->ptr1);
}

/*************** a set of transitions ***************************
 * Warning: This will not catch inconsistencies in the specification.
 * Clay will catch those later.
 */

/* transitions    :  logic_OR ';' | '{' logic_list '}' ;
 * logic_list     : logic_OR ';' | logic_OR ';' logic_list ;
 */
void Semantics::transitions(TreeNode* nodePtr) {
  err("trans\n");
  if (nodePtr->kind == transitions__1)
    logic_OR(nodePtr->ptr1);
  else
    logic_list(nodePtr->ptr1);
}

void Semantics::logic_list(TreeNode* nodePtr) {
  err("log list\n");
  logic_OR(nodePtr->ptr1);
  if (nodePtr->kind == transitions__2)
    logic_list(nodePtr->ptr2);
}

/* logic_OR       : logic_AND | logic_OR OROR logic_AND ;
 * logic_AND      : equality_exp | logic_AND ANDAND equality_exp ;
 */
void Semantics::logic_OR(TreeNode* nodePtr) {
  err("  log or");
  if (nodePtr->kind == logic_OR__1)
    logic_AND(nodePtr->ptr1);
  else {
    logic_OR(nodePtr->ptr1);
    logic_AND(nodePtr->ptr2);
  }
}

/* logic_AND      : logic_NOT | logic_AND ANDAND logic_NOT */
void Semantics::logic_AND(TreeNode* nodePtr) {
  err("  log and");
  if (nodePtr->kind == logic_AND__1)
    logic_NOT(nodePtr->ptr1);
  else {
    logic_AND(nodePtr->ptr1);
    logic_NOT(nodePtr->ptr2);
  }
}
/* 40) logic_NOT : '!' logic_end | logic_end */
void Semantics::logic_NOT(TreeNode* nodePtr) {
  if (nodePtr->kind == logic_NOT__1) {
    logic_end(nodePtr->ptr1);
  }
  else if (nodePtr->kind == logic_NOT__2) {
    logic_end(nodePtr->ptr1);
  }
  else treeError("logic_NOT");
}


/* 41) logic_end : '(' logic_OR ')' | rel_exp ;*/
void Semantics::logic_end(TreeNode* nodePtr) {
  if (nodePtr->kind == logic_end__1) {
    logic_OR(nodePtr->ptr1);
  }
  else if (nodePtr->kind == logic_end__2) {
    rel_exp(nodePtr->ptr1);
  }
  else treeError("logic_end");
}

/* 42) rel_exp : add_exp relop add_exp */
void Semantics::rel_exp(TreeNode* nodePtr) {
  if (nodePtr->kind == rel_exp__1) {
    add_exp(nodePtr->ptr1);
    relop(nodePtr->ptr2);
    add_exp(nodePtr->ptr3);
  }
  else treeError("rel_exp");
}


/* 43) relop :   '>' | GEQ | '<' | LEQ | EQEQ | NEQ */
void Semantics::relop(TreeNode* nodePtr) {
  if (nodePtr->kind == relop__1) {
  }
  else if (nodePtr->kind == relop__2) {
  }
  else if (nodePtr->kind == relop__3) {
  }
  else if (nodePtr->kind == relop__4) {
  }
  else if (nodePtr->kind == relop__5) {
  }
  else if (nodePtr->kind == relop__6) {
  }
  else treeError("relop");
}

/* 44) add_exp : mul_exp | add_exp addop mul_exp */
void Semantics::add_exp(TreeNode* nodePtr) {
  if (nodePtr->kind == add_exp__1) {
    mul_exp(nodePtr->ptr1);
  }
  else if (nodePtr->kind == add_exp__2) {
    add_exp(nodePtr->ptr1);
    addop(nodePtr->ptr2);
    mul_exp(nodePtr->ptr3);
  }
  else treeError("add_exp");
}

/* 45) addop : '+' | '-' */
void Semantics::addop(TreeNode* nodePtr) {
  if (nodePtr->kind == addop__1) {
  }
  else if (nodePtr->kind == addop__2) {
  }
  else treeError("addop");
}

/* 46) mul_exp */
void Semantics::mul_exp(TreeNode* nodePtr) {
  if (nodePtr->kind == mul_exp__1) {
    exp_end(nodePtr->ptr1);
  }
  else if (nodePtr->kind == mul_exp__2) {
    mul_exp(nodePtr->ptr1);
    exp_end(nodePtr->ptr2);
  }
  else treeError("mul_exp");
}

// exp_end : '(' add_exp ')' | ID | num | BOOL | OLD '(' ID ')' | VALUE | COUNT
void Semantics::exp_end(TreeNode* nodePtr) {
  err("  log end\n");
  if (nodePtr->kind == exp_end__1) // '(' add_exp ')'
    add_exp(nodePtr->ptr1);

  if (nodePtr->kind == exp_end__6)  { // VALUE
    // ERROR if any fields were declared
    scope->startIterator();
    SemanticRec* nextRec;
    while (scope->moreToIterate()) {
      nextRec = scope->nextIteratedSR();
      if(nextRec->kind == FIELDINFO){
	cerr << "Line " << nodePtr->line << ": " 
	     << "value cannot be used when fields have been declared." << endl;
	errCount++;
      }
    }
  }

  else if (nodePtr->kind == exp_end__2) {       // ID
    SemanticRec* tempRec = scope->lookup(nodePtr->ptr1->strval);

    // ERROR if the ID isn't declared 
    if (tempRec == NULL) {
      cerr << "Line " << nodePtr->ptr1->line << ": " 
	   << nodePtr->ptr1->strval 
	   << " is undefined" 
	   << endl;
      errCount++;
    }

    // ERROR if the ID isnt a field or state or enum
    if (tempRec->kind != FIELDINFO && tempRec->kind != STATEINFO
	&& tempRec->kind != ENUMINFO) {
      cerr << "Line " << nodePtr->ptr1->line << ": " 
	   << nodePtr->ptr1->strval 
	   << " is not a field, state, or enum value." 
	   << endl;
      errCount++;
    }

    // ERROR if this is a field pre-condition on a read in a readable port
    if (tempRec->kind == FIELDINFO
	&& io=="before" && mode!=read_write__2 
	&& currPort->access != read_write__2) {
      cerr << "Line " << nodePtr->ptr1->line << ": " 
	   << "Preconditions on reads cannot refer to fields." 
	   << endl;
      errCount++;
    }
    
  }

  else if (nodePtr->kind == exp_end__5) { // OLD '(' ID ')'
    SemanticRec* tempRec = scope->lookup(nodePtr->ptr1->strval);

    // ERROR if the ID isn't declared
    if (tempRec == NULL) {
      cerr << "Line " << nodePtr->ptr1->line << ": " 
	   << nodePtr->ptr1->strval 
	   << " is undefined" 
	   << endl;
      errCount++;
    }

    // ERROR if the ID isnt a state
    if (tempRec->kind != STATEINFO) {
      cerr << "Line " << nodePtr->ptr1->line << ": " 
	   << nodePtr->ptr1->strval 
	   << " is not a state." 
	   << endl;
      errCount++;
    }

    // ERROR if there is an OLD in a precondition
    if (io=="before") {
      cerr << "Line " << nodePtr->ptr1->line << ": " 
	   << "Preconditions cannot refer to old states." 
	   << endl;
      errCount++;
    }
  }

  // nothing for value (6)

  else if (nodePtr->kind == exp_end__7) {
    // ERROR if this port is not repeated
    if (currPort->type != 1) { /* 1: repeated */
      cerr << "Line " << currPort->line << ": "
	   << "Conditions may only refer to count if the port is repeated."
	   << endl;
      errCount++;
    }
  }
}

  /****************************************************************************/
  /*                                                                          */
  /* helper routines                                                          */
  /*                                                                          */
  /****************************************************************************/

  // if one range is complete outside the other, there is no bit sharing conflict
  bool Semantics::doBitsConflict(SemanticRec* newRec, SemanticRec* currRec){
    if((newRec->range.getRange(0).getLow() < currRec->range.getRange(0).getLow() 
	&& newRec->range.getRange(0).getHigh() < currRec->range.getRange(0).getLow())
       ||
       (newRec->range.getRange(0).getLow() > currRec->range.getRange(0).getHigh() 
	&& newRec->range.getRange(0).getHigh() > currRec->range.getRange(0).getHigh()))
      return false; //ranges do no coincide
    else
      return true;
  }

  // Make a new scope and push it on the symbol table.
  void Semantics::pushScope() {
    SymbolTable * tempScope = new SymbolTable(); // Make a new scope
    tempScope->prevScope = scope;              // Set the prev scope
    scope = tempScope;                         // Point to the scope
  }

  //remove current scope
  void Semantics::popScope() {
    scope = scope->prevScope;
  }

  //returns the max value a bit range is able to hold
  double Semantics::maxBitValue(SemanticRec* currRec){
    double bitDiff = currRec->range.getRange(0).getHigh() 
      - currRec->range.getRange(0).getLow();
    return pow(2.0, bitDiff + 1) - 1;
  }

  //returns true if a name already exists, otherwise false 
  bool Semantics::nameExists(string name, vector<string> & names){
    int i = 0;
    while(i < names.size()){
      if(name == names[i])
	return true;

      i++;
    }

    return false;
  }

  // is port "number" in read/write mode (r,w,rw) unique?
  bool Semantics::isPortUnique() {
    SemanticRec* nextRec;
    scope->startIterator();

    // get stateCount, stateSizes, stateNames
    while (scope->moreToIterate()) {
      nextRec = scope->nextIteratedSR();
      if (nextRec->kind == PORTINFO && nextRec->num == currPort->num // same port num
	  && !(nextRec->key == currPort->key)                        // not this port
	  && (nextRec->access == currPort->access ||                 // same access
	      nextRec->access == read_write__3 ||                    // it's rw
	      currPort->access == read_write__3)) {                  // this port rw
	return false;
      }
    }
    return true;
  }
