/* 
 * Laddie to Clay Compiler
 * Author: Lea Wittie and Derrin Pierret
 * Copyright (c) 2007 Bucknell University
 *
 * Permission is hereby granted, free of charge, to any individual or
 * institution obtaining a copy of this software and associated
 * documentation files (the "Software"), to use, copy, modify, and
 * distribute without restriction, provided that this copyright and
 * permission notice is maintained, intact, in all copies and supporting
 * documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL BUCKNELL UNIVERSITY BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
using namespace std;
#include <iostream>
#include <string>
#include "compiler.h"
#include "support.h"
#include "Treewalk.h"
#include "Semantics.h"
#include "Generate.h"
#include <fstream>
#include <sstream>
#include <cmath>

string itoa(int number){
    std::stringstream ss;
    std::string str;
    ss << number;
    ss >> str;
    return str;
}

// generate an N bit mask where N is finish-start
unsigned int genReadMask(int startBit, int finishBit){
  unsigned int mask = 0;

    for(int i = 0 ; i<=(finishBit - startBit) ; i++){
      // cout<<"read masking"<<endl;
	    mask += (unsigned int)pow(2.0,i);
    }

    return mask;
}

unsigned int genWriteMask(int startBit, int finishBit){
     unsigned int mask = 0;

    for(int i = startBit ; i<=finishBit ; i++){     
      //   cout<<"write masking"<<endl;
	    mask += (unsigned int)pow(2.0,i);
    }

    return mask;
}


/****************************************************************************/
/*                                                                          */
/* ClayNameGen (generates names for clay variables)                         */
/*   (names are sent to bit ranges first, then to user defined states)      */
/*                                                                          */
/****************************************************************************/


//clay names generated from this class include upper case letter followed by
//a number 0-9... so 260 possible names, which should be more than enough
//for now, and can be easily changed if need be

ClayNameGen::ClayNameGen(){
  currChar = 65;
  currNumber = 0;
}

void ClayNameGen::genNextName(){
  if(currChar == 90){ //last letter
    currChar = 65;
    currNumber++;
  }
  else{
    currChar++;
  }
}

string ClayNameGen::getCurrName(){
    string name = (char)(currChar) + itoa(currNumber);
  return name;
}

void ClayNameGen::reset(){
  currChar = 65;
  currNumber = 0;
}


/****************************************************************************/
/*                                                                          */
/* Range (holds allowed range for a boundary node)                          */
/*                                                                          */
/****************************************************************************/

Range::Range(){
    hi = -1;
    low = -1;
}

Range::Range(int inLow, int inHi){
    hi = inHi;
    low = inLow;
}

bool Range::isInRange(int value){
    return (value <= hi && value >= low);      
}

void Range::setRange(int inLow, int inHi){
    low = inLow;
    hi = inHi;
}

int Range::getHigh(){
    return hi;
}

int Range::getLow(){
    return low;
}

bool Range::isNull(){
    return (hi == -1 && low == -1);
}

/****************************************************************************/
/*                                                                          */
/* Boundary (holds allowed ranges for a user defined field)                 */
/*                                                                          */
/****************************************************************************/

Boundary::Boundary(){

}

void Boundary::add(int num1){ //single number allowed
    add(num1, num1);
}

void Boundary::add(int num1, int num2){ //range allowed
    if(num1 <= num2)
	set.push_back(Range(num1, num2));
    else
	set.push_back(Range(num2, num1));
}

bool Boundary::isInBounds(int value){
    if(set.size() == 0)   //empty set allows all possible values
	return true;

    for(int i = 0 ; i < set.size() ; i++){
	if(set.at(i).isInRange(value))
	    return true;
    }
    return false;
}

bool Boundary::isEmpty() {
    return (set.size() == 0);
}

Range Boundary::getRange(int index){
    return set.at(index);
}
 
int Boundary::getSize(){
    return set.size();
}

/****************************************************************************/
/*                                                                          */
/* Treewalk                                                                 */
/*                                                                          */
/****************************************************************************/
void Treewalk::treeError(char* error) {
    fprintf(stderr, "Error walking the tree. Node %s.\n", error);
    exit(1);
}

/****************************************************************************/
/*                                                                          */
/* Generate                                                                 */
/*                                                                          */
/****************************************************************************/
void Generate::treeError(char* error) {
    fprintf(stderr, "Error walking the tree. Node %s.\n", error);
    exit(1);
}

/****************************************************************************/
/*                                                                          */
/* Semantics                                                                */
/*                                                                          */
/****************************************************************************/
void Semantics::treeError(char* error) {
    fprintf(stderr, "Error walking the tree. Node %s.\n", error);
    exit(1);
}

/****************************************************************************/
/*                                                                          */
/* SemanticRec                                                              */
/*                                                                          */
/****************************************************************************/
SemanticRec::SemanticRec(int inKind, int inLine) {
  kind = inKind;
  line = inLine;
  lineDefined = -1;  // not defined currently
}

SemanticRec::SemanticRec(int inKind, int inLine, string inKey) {
  kind = inKind;
  line = inLine;
  key = inKey;
  lineDefined = -1;  // not defined currently
}

SemanticRec::~SemanticRec(){
}

/****************************************************************************/
/*                                                                          */
/* HashNode                                                                 */
/*                                                                          */
/****************************************************************************/
HashNode::HashNode() {
     theSemRec = NULL;
     nextNode = NULL;
}

HashNode::~HashNode() { 
    delete theSemRec;
    delete nextNode;
}

/****************************************************************************/
/*                                                                          */
/* SymbolTable                                                              */
/*                                                                          */
/****************************************************************************/
SymbolTable::SymbolTable() { 
    int i;
    for(i=0; i<M; i++) theTable[i] = NULL; // Make the pointers NULL
    prevScope = NULL;                        // Previous symbol table
}

// Delete the symbol table (and its hash table)
SymbolTable::~SymbolTable() { 
    int i;
    for (i=0; i<M; i++) {                        // Destroy old hash table
	delete theTable[i];
    }
}

int SymbolTable::hashFunct(string theName) {
     int len,i, value;
     value = 0;
     
     len = theName.length();
     for (i=0;i<len;i++)
	  value += theName[i];

     return (value % M);
}

void SymbolTable::insert(SemanticRec* theRec) {
  insert(theRec->key, theRec);
}

// Insert a record into the symbol table, keyed by name
void SymbolTable::insert(string theName, SemanticRec* theRec) {
     int slot = hashFunct(theName);          // Hash the key.
     
     HashNode* box = new HashNode();         // Make a node for it.
     theRec->key = theName;
     box->key = theName;
     box->theSemRec = theRec;
     box->nextNode = NULL;
     
     // if it's first, put it there
     if (theTable[slot] == NULL) 
	 theTable[slot] = box;

     // else find the predecessor and put it on the list
     else {
	 HashNode* tmp;
	 for (tmp = theTable[slot]; 
	      tmp->nextNode != NULL;
	      tmp = tmp->nextNode);
	 tmp->nextNode = box;
     }
}

// Lookup a symbol table record, keyed by name
SemanticRec* SymbolTable::lookup(string theName) {
    int slot;
    SemanticRec* tempRec;
    HashNode* tempNode;
    
    slot = hashFunct(theName);               // Hash the key.

    // Look for the node locally
    tempNode = theTable[slot];
     while((tempNode != NULL) && (tempNode->key != theName))
	  tempNode = tempNode->nextNode;

     // If not found, search the previous scope or give up
     if (tempNode == NULL) {
	 if (prevScope != NULL)
	     tempRec = prevScope->lookup(theName);
	 else 
	     tempRec = NULL;
     }

     // If found, retrieve the Semantic Record
     else
	 tempRec = tempNode->theSemRec;

     return tempRec;
}


/****************************************************************************/
/*                                                                          */
/* TreeNode                                                                 */
/*                                                                          */
/****************************************************************************/
TreeNode::TreeNode() { }

TreeNode::TreeNode(int k) {
    kind = k;
}

TreeNode::TreeNode(int k, int lineno) {
    kind = k;
    line = lineno;
}

/****************************************************************************/
/*                                                                          */
/* Token Iterator                                                           */
/*                                                                          */
/****************************************************************************/
// If there are any SemanticRecs in the table, this will be at the first one
void SymbolTable::startIterator() {
    currentSlot = 0;
    currentHN = theTable[0];
    while(currentSlot<M-1 && currentHN == NULL) {
        currentSlot++;
        currentHN = theTable[currentSlot];
    }
}

// If there are more SemanticRecs it will be positive, else NULL
bool SymbolTable::moreToIterate() {
    return (currentHN != NULL);
}


// Find the next SemanticRec (or null)
SemanticRec* SymbolTable::nextIteratedSR() {
    // The next one we havent returned
    HashNode* thisOne = currentHN;
    // And set up for the one after it

    // Either we stay at the same slot
    if (currentHN->nextNode != NULL) {
	currentHN = currentHN->nextNode;
    }

    // Or we find the next slot with stuff in it (or we leave it on the last
    // (null) slot)
    else {    
	do {
	    currentSlot++;
	    if (currentSlot < M) currentHN = theTable[currentSlot];
	    //fprintf(stderr, "..%d", currentSlot);
	    }
	while(currentSlot<M && currentHN == NULL);
    }

    if (currentSlot == M) // nothing more to see here
	currentHN = NULL;

    return thisOne->theSemRec;
}
